<?php
/*
 * @package     RadicalMart Package
 * @subpackage  plg_radicalmart_payment_mandarin
 * @version     __DEPLOY_VERSION__
 * @author      Delo Design - delo-design.ru
 * @copyright   Copyright (c) 2022 Delo Design. All rights reserved.
 * @license     GNU/GPL license: https://www.gnu.org/copyleft/gpl.html
 * @link        https://delo-design.ru/
 */

defined('_JEXEC') or die;

use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Factory;
use Joomla\CMS\Http\Http;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Registry\Registry;

class plgRadicalMart_PaymentMandarin extends CMSPlugin
{
	/**
	 * Loads the application object.
	 *
	 * @var  CMSApplication
	 *
	 * @since   __DEPLOY_VERSION__
	 */
	protected $app = null;

	/**
	 * Loads the database object.
	 *
	 * @var  JDatabaseDriver
	 *
	 * @since  __DEPLOY_VERSION__
	 */
	protected $db = null;

	/**
	 * Affects constructor behavior.
	 *
	 * @var  boolean
	 *
	 * @since   __DEPLOY_VERSION__
	 */
	protected $autoloadLanguage = true;

	/**
	 * Payment method params.
	 *
	 * @var  Registry
	 *
	 * @since   __DEPLOY_VERSION__
	 */
	protected $_paymentMethodParams = null;

	/**
	 * Prepare order shipping method data.
	 *
	 * @param   string  $context   Context selector string.
	 * @param   object  $method    Method data.
	 * @param   array   $formData  Order form data.
	 * @param   array   $products  Order products data.
	 * @param   array   $currency  Order currency data.
	 *
	 * @throws  Exception
	 *
	 * @since  __DEPLOY_VERSION__
	 */
	public function onRadicalMartGetPaymentMethods($context, $method, $formData, $products, $currency)
	{
		// Set disabled
		$method->disabled = false;

		// Clean secret param
		$method->params->set('merchant_id', '');
		$method->params->set('secret', '');
		$method->params->set('client_id', '');
		$method->params->set('client_secret', '');

		// Set order
		$method->order              = new stdClass();
		$method->order->id          = $method->id;
		$method->order->title       = $method->title;
		$method->order->code        = $method->code;
		$method->order->description = $method->description;
		$method->order->price       = array();
	}

	/**
	 * Check can order pay.
	 *
	 * @param   string  $context  Context selector string.
	 * @param   object  $order    Order Item data.
	 *
	 * @throws  Exception
	 *
	 * @return boolean True if can pay, False if not.
	 *
	 * @since  __DEPLOY_VERSION__
	 */
	public function onRadicalMartCheckOrderPay($context, $order)
	{
		// Check order payment method
		if (empty($order->payment)
			|| empty($order->payment->id)
			|| empty($order->payment->plugin)
			|| $order->payment->plugin !== 'mandarin') return false;


		// Check method params
		$params = $this->getPaymentMethodParams($order->payment->id);
		if (empty($params->get('merchant_id')) || empty($params->get('secret'))
			|| empty($params->get('client_id')) || empty($params->get('client_secret')))
		{
			return false;
		}

		// Check order status
		if (empty($order->status->id) || !in_array($order->status->id, $params->get('payment_available', array())))
		{
			return false;
		}

		return true;
	}

	/**
	 * Method to create transaction in RadicalMart.
	 *
	 * @param   object    $order   Order data object.
	 * @param   Registry  $params  RadicalMart component params.
	 * @param   array     $links   RadicalMart plugin links.
	 *
	 * @throws  Exception
	 *
	 * @return  array  Payment redirect data on success.
	 *
	 * @since   __DEPLOY_VERSION__
	 */
	public function onRadicalMartPay($order, $links, $params)
	{
		$result = array(
			'pay_instant' => true,
			'link'        => false,
		);

		// Check order payment method
		if (empty($order->payment)
			|| empty($order->payment->id)
			|| empty($order->payment->plugin)
			|| $order->payment->plugin !== 'mandarin') return $result;

		// Get method params
		$params = $this->getPaymentMethodParams($order->payment->id);
		if (empty($params->get('merchant_id')) || empty($params->get('secret'))
			|| empty($params->get('client_id')) || empty($params->get('client_secret')))
		{
			throw new Exception(Text::_('PLG_RADICALMART_PAYMENT_MANDARIN_ERROR_EMPTY_PAYMENTS_METHOD_PARAMS'), 500);
		}

		// Check order status
		if (empty($order->status->id) || !in_array($order->status->id, $params->get('payment_available', array())))
		{
			throw new Exception(Text::_('PLG_RADICALMART_PAYMENT_MANDARIN_ERROR_PAYMENT_NOT_AVAILABLE'), 500);
		}

		// Prepare data
		$data = array(
			'payment'      => array(
				'action'          => 'pay',
				'orderId'         => $order->number,
				'price'           => $order->total['final'],
				'orderActualTill' => Factory::getDate('+ 3 day')->format('Y-m-d h:i:sP')
			),
			'customerInfo' => array(),
			'urls'         => array(
				'return'   => $links['success'] . '/' . $order->number,
				'callback' => $links['callback']
			),
		);
		if (!empty($order->contacts['email']))
		{
			$data['customerInfo']['email'] = $order->contacts['email'];
		}
		if (!empty($order->contacts['phone']))
		{
			$data['customerInfo']['phone'] = $order->contacts['phone'];
		}

		// Add fiscalInformation
		if ($params->get('fiscal'))
		{
			$data['fiscalInformation'] = array(
				'taxationSystem' => $params->get('taxation_system', 'Common'),
				'items'          => array(),
			);

			// Prepare products
			foreach ($order->products as $product)
			{
				$data['fiscalInformation']['items'][] = array(
					'description' => $product->title,
					'quantity'    => $product->order['quantity'],
					'totalPrice'  => $product->order['sum_final'],
					'vat'         => $params->get('vat', 'None')
				);
			}

			// Add shipping
			if (!empty($order->shipping) && !empty($order->shipping->order)
				&& !empty($order->shipping->order->price)
				&& (!empty($order->shipping->order->price['base']) || !empty($order->shipping->order->price['final']))
			)
			{
				$shipping = $order->shipping;

				$data['fiscalInformation']['items'][] = array(
					'description' => (!empty($shipping->order->title)) ? $shipping->order->title : $shipping->title,
					'quantity'    => 1,
					'totalPrice'  => (!empty($shipping->order->price['base'])) ? $shipping->order->price['base']
						: $shipping->order->price['final'],
					'vat'         => $params->get('vat', 'None')
				);
			}
		}

		try
		{
			$response = $this->createTransaction($data, $params->get('merchant_id'), $params->get('secret'));

			$result['link'] = $response->get('userWebLink');

			return $result;
		}
		catch (Exception $e)
		{
			throw new Exception('Mandarin: ' . $e->getMessage(), $e->getCode());
		}
	}

	/**
	 * Method to set order pay status after payment.
	 *
	 * @param   array                    $input   Input data.
	 * @param   RadicalMartModelPayment  $model   RadicalMart model.
	 * @param   Registry                 $params  RadicalMart params.
	 *
	 * @since  __DEPLOY_VERSION__
	 */
	public function onRadicalMartCallback($input, $model, $params)
	{
		// Add logger
		Log::addLogger(array(
			'text_file'         => 'plg_radicalmart_payment_mandarin.php',
			'text_entry_format' => "{DATETIME}\t{CLIENTIP}\t{MESSAGE}\t{PRIORITY}"),
			Log::ALL, array('plg_radicalmart_payment_mandarin'));

		try
		{
			// Check transaction
			if (empty($input['transaction']))
			{
				throw new Exception(Text::_('PLG_RADICALMART_PAYMENT_MANDARIN_ERROR_TRANSACTION_NOT_FOUND'));
			}

			// Get order
			if (empty($input['orderId']))
			{
				throw new Exception(Text::_('COM_RADICALMART_EXPRESS_ERROR_ORDER_NOT_FOUND'));
			}
			if (!$order = $model->getOrder($input['orderId']))
			{
				$messages = array();
				foreach ($model->getErrors() as $error)
				{
					$messages[] = ($error instanceof Exception) ? $error->getMessage() : $error;
				}
				if (empty($messages)) $messages[] = Text::_('COM_RADICALMART_EXPRESS_ERROR_ORDER_NOT_FOUND');

				throw new Exception(implode(PHP_EOL, $messages), 404);
			}

			// Check order payment method
			if (empty($order->payment)
				|| empty($order->payment->id)
				|| empty($order->payment->plugin)
				|| $order->payment->plugin !== 'mandarin') $this->setResponse();

			// Get method params
			$params = $this->getPaymentMethodParams($order->payment->id);
			if (empty($params->get('merchant_id')) || empty($params->get('secret'))
				|| empty($params->get('client_id')) || empty($params->get('client_secret')))
			{
				throw new Exception(Text::_('PLG_RADICALMART_PAYMENT_MANDARIN_ERROR_EMPTY_PAYMENTS_METHOD_PARAMS'), 500);
			}

			// Get transaction data
			$transaction = $this->getTransaction($input['transaction'], $params->get('client_id'),
				$params->get('client_secret'));

			// Set order status
			$newStatus = (int) $params->get('paid_status');
			if ($newStatus && $newStatus !== (int) $order->status->id && $transaction->status === 'success')
			{
				$model->addLog($order->id, 'mandarin_paid', array(
					'plugin'         => 'mandarin',
					'group'          => 'radicalmart_payment',
					'transaction'    => $input['transaction'],
					'transaction_id' => $transaction->id,
					'user_id'        => -1
				));

				if (!$model->updateStatus($order->id, $newStatus, false, -1))
				{
					$messages = array();
					foreach ($model->getErrors() as $error)
					{
						$messages[] = ($error instanceof Exception) ? $error->getMessage() : $error;
					}
					throw new Exception(implode(PHP_EOL, $messages), 500);
				}
			}

			return $this->setResponse();
		}
		catch (Exception $e)
		{
			Log::add($e->getMessage(), Log::ERROR, 'plg_radicalmart_payment_mandarin');

			return $this->setResponse();
		}
	}
	/**
	 * Method to display logs.
	 *
	 * @param   string  $context  Context selector string.
	 * @param   array   $log      Log data.
	 *
	 * @since  __DEPLOY_VERSION__
	 */
	public function onRadicalMartGetOrderLogs($context = null, &$log = array())
	{
		if ($log['action'] === 'mandarin_paid')
		{
			$log['action_text'] = Text::_('PLG_RADICALMART_PAYMENT_MANDARIN_LOGS_MANDARIN_PAID');
			if (!empty($log['transaction_id']))
			{
				$log['message'] = Text::sprintf('PLG_RADICALMART_PAYMENT_MANDARIN_LOGS_MANDARIN_PAID_MESSAGE',
					$log['transaction'], $log['transaction_id']);
			}
		}
	}

	/**
	 * Method to get payment method params.
	 *
	 * @param   int  $pk  Payment method id.
	 *
	 * @return Registry Payment method params
	 *
	 * @since  __DEPLOY_VERSION__
	 */
	protected function getPaymentMethodParams($pk = null)
	{
		$pk = (int) $pk;
		if (empty($pk)) return new Registry();

		if ($this->_paymentMethodParams === null) $this->_paymentMethodParams = array();
		if (!isset($this->_paymentMethodParams[$pk]))
		{
			$db                              = $this->db;
			$query                           = $db->getQuery(true)
				->select('params')
				->from($db->quoteName('#__radicalmart_payment_methods'))
				->where('id = ' . $pk);
			$this->_paymentMethodParams[$pk] = ($result = $db->setQuery($query, 0, 1)->loadResult())
				? new Registry($result) : new Registry();
		}

		return $this->_paymentMethodParams[$pk];
	}

	/**
	 * Method to create transaction in Mandarin.
	 *
	 * @param   array   $data        transaction
	 * @param   int     $merchantId  Merchant id.
	 * @param   string  $secret      Merchant secret.
	 *
	 * @throws Exception
	 *
	 * @return Registry Request result.
	 *
	 * @since  __DEPLOY_VERSION__
	 */
	protected function createTransaction($data = array(), $merchantId = null, $secret = null)
	{
		$merchantId = (int) trim($merchantId);
		$secret     = trim($secret);
		if (empty($merchantId) || empty($secret))
		{
			throw new Exception(Text::_('PLG_RADICALMART_PAYMENT_MANDARIN_ERROR_EMPTY_PAYMENTS_METHOD_PARAMS'));
		}

		$requestId = time() . '_' . microtime(true) . '_' . rand();
		$hash      = hash('sha256', $merchantId . '-' . $requestId . '-' . $secret);
		$xAuth     = $merchantId . '-' . $hash . '-' . $requestId;

		$data = (new Registry($data))->toString('json', array('bitmask' => JSON_UNESCAPED_UNICODE));

		$http = new Http();
		$http->setOption('transport.curl', array(CURLOPT_SSL_VERIFYHOST => 0, CURLOPT_SSL_VERIFYPEER => 0));
		$response = $http->post('https://secure.mandarinpay.com/api/transactions', $data,
			array('X-Auth' => $xAuth, 'Content-Type' => 'application/json'));

		$context = (!empty($response->body)) ? new Registry($response->body) : false;
		if ($response->code !== 200 || !$context || $context->get('errorCode'))
		{
			$message = ($context) ? $context->get('error')
				: preg_replace('#^[0-9]*\s#', '', $response->headers['Status']);
			$code    = ($context) ? $context->get('errorCode') : $response->code;

			throw new Exception($message, $code);
		}

		return $context;
	}

	/**
	 * Method to get transaction from Mandarin.
	 *
	 * @param   string  $transaction   Transaction identifier.
	 * @param   string  $clientId      Client id.
	 * @param   string  $clientSecret  Client secret.
	 *
	 * @throws Exception
	 *
	 * @return object Transaction data.
	 *
	 * @since  __DEPLOY_VERSION__
	 */
	protected function getTransaction($transaction = null, $clientId = null, $clientSecret = null)
	{
		$transaction = trim($transaction);
		if (empty($transaction))
		{
			throw new Exception(Text::_('PLG_RADICALMART_PAYMENT_MANDARIN_ERROR_TRANSACTION_NOT_FOUND'));
		}

		$clientId     = trim($clientId);
		$clientSecret = trim($clientSecret);
		if (empty($clientId) || empty($clientSecret))
		{
			throw new Exception(Text::_('PLG_RADICALMART_PAYMENT_MANDARIN_ERROR_EMPTY_PAYMENTS_METHOD_PARAMS'));
		}

		// Get token
		$http = new Http();
		$http->setOption('transport.curl', array(CURLOPT_SSL_VERIFYHOST => 0, CURLOPT_SSL_VERIFYPEER => 0));
		$response = $http->post('https://accounts.mandarinbank.com/oauth/token/', array(
			'grant_type'    => 'client_credentials',
			'client_id'     => $clientId,
			'client_secret' => $clientSecret,
			'scope'         => 'transactions.read'
		));

		$context = (!empty($response->body)) ? new Registry($response->body) : false;
		if ($response->code !== 200 || !$context || $context->get('errorCode'))
		{
			$message = ($context) ? $context->get('error')
				: preg_replace('#^[0-9]*\s#', '', $response->headers['Status']);
			$code    = ($context) ? $context->get('errorCode') : $response->code;

			throw new Exception($message, $code);
		}

		$token        = $context->get('access_token');
		$counter      = 0;
		$object       = false;
		while (true)
		{
			$counter++;
			sleep(10);

			$error        = false;
			$errorMessage = '';
			$errorCode    = '';

			// Get transaction
			$http = new Http();
			$http->setOption('transport.curl', array(CURLOPT_SSL_VERIFYHOST => 0, CURLOPT_SSL_VERIFYPEER => 0));
			$response = $http->get('https://api.psp.io/payment-gateway/v3/transactions?filter_by=cs1%3D' . $transaction,
				array('Authorization' => 'Bearer ' . $token));

			$context = (!empty($response->body)) ? new Registry($response->body) : false;
			if ($response->code !== 200 || !$context || $context->get('status')->success === 'false')
			{
				$message = ($context) ? $context->get('status')->errors[0]->message
					: preg_replace('#^[0-9]*\s#', '', $response->headers['Status']);
				$code    = ($context) ? $context->get('errorCode') : $response->code;

				$error        = true;
				$errorMessage = $message;
				$errorCode    = $code;
			}

			if (!$error)
			{
				$transactions = $context->get('transactions');
				if (!empty($transactions[0]))
				{
					$errorMessage = '';
					$errorCode = '';
					$object = $transactions[0];
					break;
				}
			}
			if ($counter === 10) break;
		}

		if (!empty($errorMessage))
		{
			throw new Exception($errorMessage, $errorCode);
		}

		if (empty($object))
		{
			throw new Exception(Text::_('PLG_RADICALMART_PAYMENT_MANDARIN_ERROR_TRANSACTION_NOT_FOUND'));
		}

		return $object;
	}

	/**
	 * Method to set Mandarin respoce.
	 *
	 * @since  __DEPLOY_VERSION__
	 */
	protected function setResponse()
	{
		echo 'OK';
		$this->app->close(200);

		return true;
	}
}