<?php
/*
 * @package     mod_msg_rmf
 * @version     1.0.0
 * @author      Vladislav Tsygankov
 * @copyright   Copyright (c) 2023 MSGroup. All rights reserved.
 * @license     GNU/GPL license: https://www.gnu.org/copyleft/gpl.html
 * @link        https://msgru.com/
 */

namespace MSGJoomla\Module\RMFilter\Site\Adapter\ProductField;

// no direct access
\defined('JPATH_PLATFORM') or die;

use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\CMS\Cache\Controller\CallbackController;
use Joomla\CMS\Cache\Exception\CacheExceptionInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\FormField;
use Joomla\CMS\Language\Multilanguage;
use Joomla\Database\DatabaseInterface;
use Joomla\Database\ParameterType;
use Joomla\Component\RadicalMart\Administrator\Helper\PriceHelper;

/**
 * Adapter for joomla field type list
 *
 * @since  1.0.0
 */
class ListFieldAdapter extends BaseFieldAdapter
{
    protected string $layout = 'modules.msg_rmf.filter.field.list';

    protected string $type = 'list';
    protected $field;
    protected bool $multiSelect = false;

    /**
     * @throws \Exception
     */
    public function setup(): void
    {
        parent::setup();
        
        $this->multiSelect = $this->field->params->get('type', 'unknown') == 'checkboxes' || $this->field->params->get('multiple', false);
    }

    public function handle(): void
    {
        parent::handle();
        if (!$this->show) return;

        $caching = $this->factory->moduleParams->get('cache') !== '0';

        /** @var CallbackController $cache */
        $cache = Factory::getContainer()->get(CacheControllerFactoryInterface::class)->createCacheController('callback', ['defaultgroup' => 'mod_msg_rmf', 'caching' => $caching]);

        $cache->setLifeTime($this->factory->moduleParams->get('cache_time', Factory::getApplication()->get('cachetime') * 60) / 60);

        if ($this->factory->moduleParams->get('show_empty_options', 2) !== 2)
        {
            try {
                $disabledOptions = $cache->get([$this, 'setOptions'], array(), md5(serialize(array($this->factory->categoryId, $this->field->id, $this->multiSelect))));
            } catch (CacheExceptionInterface $e) {
                $disabledOptions = $this->setOptions();
            }
        }

        $needHide = true;

        foreach ($this->xml->children() as $option)
        {
            $value = (string)$option->attributes()->value;

            $disabled = !empty($value) && in_array($value, $disabledOptions);

            $option->addAttribute('disabled', $disabled);

            $needHide = $disabled;
        }

        $this->show = !$needHide;
    }

    /**
     * Method to get field options with hidden or disable attr.
     *
     * @param   FormField  $field  Form field.
     * @param   int  $category_id  RadicalMart category id.
     * @param   bool  $show  set disable or hidden attribute. If true, that use disable.
     *
     * @throws  \Exception
     *
     * @return  array  Field options.
     *
     * @since  1.0.0
     */

    public function setOptions(): array
    {
        /** @var DatabaseInterface $db */
        $db = Factory::getContainer()->get(DatabaseInterface::class);
        $app      = Factory::getApplication();

        // Set published && debug state
        if ($app->input->getInt('debug', 0))
        {
            $published = [0, 1];
        }
        else
        {
            $published = 1;
        }

        $query = $db->getQuery(true);

        $subQuery = $db->getQuery(true)
            ->select('JSON_UNQUOTE(JSON_EXTRACT(fields, '.$db->quote('$."' . $this->field->alias . '"') .')) AS options')
            ->from($db->quoteName('#__radicalmart_products', 'p'))
            ->where('FIND_IN_SET(:categoryId, '.$db->quoteName('p.categories').')');

        // Filter by language state
        if (Multilanguage::isEnabled())
        {
            $subQuery->whereIn('p.language', [Factory::getApplication()->getLanguage()->getTag(), '*'],
                ParameterType::STRING);
        }

        if (is_numeric($published))
        {
            $subQuery->where($db->quoteName('p.state') . ' = :published');

            $query->bind(':published', $published, ParameterType::INTEGER);
        }
        elseif (is_array($published))
        {
            $subQuery->whereIn('p.state', $published);
        }

        $query
            ->from('(' . (string) $subQuery . ') ' . $db->quoteName('o'))
            ->bind(':categoryId', $this->factory->categoryId, ParameterType::INTEGER);

        if($this->multiSelect) {
            $query
                ->select('DISTINCT value')
                ->from('JSON_TABLE('.$db->quoteName('o.options').', "$[*]" columns (value varchar(255) PATH "$")) ' . $db->quoteName('t'));
        } else {
            $query
                ->select('DISTINCT ' . $db->quoteName('o.options'));
        }

        try {
            $actualOptions = $db->setQuery($query)->loadColumn();
        } catch (\Exception $e) {
            $actualOptions = array();
        }
        $disabled = array();

        foreach ($this->xml->children() as $option)
        {
            $value = (string)$option->attributes()->value;

            if (!in_array($value, $actualOptions)) {
                $disabled[] = $value;
            }
        }

        return $disabled;
    }
}