<?php
/*
 * @package     mod_msg_rmf
 * @version     1.0.4
 * @author      Vladislav Tsygankov
 * @copyright   Copyright (c) 2023 MSGroup. All rights reserved.
 * @license     GNU/GPL license: https://www.gnu.org/copyleft/gpl.html
 * @link        https://msgru.com/
 */

namespace MSGJoomla\Module\RMFilter\Site\Field;

// phpcs:disable PSR1.Files.SideEffects
\defined('JPATH_PLATFORM') or die;
// phpcs:enable PSR1.Files.SideEffects

use Joomla\CMS\Factory;
use Joomla\CMS\Filesystem\Path;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Form\FormFactoryInterface;
use Joomla\CMS\Form\FormField;
use Joomla\CMS\Language\Text;
use Joomla\Registry\Registry;
use Joomla\Component\RadicalMart\Administrator\Helper\ParamsHelper;
use MSGJoomla\Module\RMFilter\Site\Helper\FilterHelper;

/**
 * The Field to load the form inside current form
 *
 * @Example with all attributes:
 *  <field name="field-name" type="sortablefields"
 *      layout="joomla.form.field.subform.repeatable-table" groupByFieldset="false" component="com_example" client="site"
 *      label="Field Label" description="Field Description" />
 *
 * @since  1.0.0
 */
class SortablefieldsField extends FormField
{
    /**
     * The form field type.
     * @var    string
     * @since  1.0.0
     */
    protected $type = 'Sortablefields';

    /**
     * Form source
     * @var string
     */
    protected $formsource;

    /**
     * Layout to render the form
     * @var  string
     * @since  1.0.0
     */
    protected $layout = 'modules.msg_rmf.form.field.sortablefields';

    /**
     * Whether group subform fields by it`s fieldset
     * @var boolean
     * @since  1.0.0
     */
    protected $groupByFieldset = false;

    /**
     * Array of available filter fields
     *
     * @var    array
     * @since  1.0.0
     */

    protected array $fields;

    /**
     * Array of available fieldsets
     *
     * @var    array
     * @since  1.0.0
     */

    protected array $_fieldsets = array();


    /**
     * Array of basic filter fields
     *
     * @var array
     * @since __DEPLAY_VERSION__
     */
    protected array $basicFields = array(
        array(
            'name' => 'categories',
            'title' => 'COM_RADICALMART_CATEGORIES',
            'type' => 'Categories',
        ),
        array(
            'name' => 'manufactures',
            'title' => 'COM_RADICALMART_MANUFACTURERS',
            'type' => 'Categories',
        ),
        array(
            'name' => 'badges',
            'title' => 'COM_RADICALMART_BADGES',
            'type' => 'Categories',
        ),
        array(
            'name' => 'in_stock',
            'title' => 'COM_RADICALMART_IN_STOCK',
            'type' => 'InStock',
        ),
        array(
            'name' => 'price',
            'title' => 'COM_RADICALMART_PRICE',
            'type' => 'Price',
        ),
    );

    protected array $correctType = array(
      'range' => 'range'
    );

    /**
     * Method to get certain otherwise inaccessible properties from the form field object.
     *
     * @param string $name The property name for which to get the value.
     *
     * @return  mixed  The property value or null.
     *
     * @since   1.0.0
     */
    public function __get($name)
    {
        switch ($name) {
            case 'formsource':
            case 'layout':
            case 'groupByFieldset':
                return $this->$name;
        }

        return parent::__get($name);
    }

    /**
     * Method to set certain otherwise inaccessible properties of the form field object.
     *
     * @param string $name The property name for which to set the value.
     * @param mixed $value The value of the property.
     *
     * @return  void
     *
     * @since   1.0.0
     */
    public function __set($name, $value)
    {
        switch ($name) {
            case 'formsource':
                $this->formsource = (string)$value;

                // Add root path if we have a path to XML file
                if (strrpos($this->formsource, '.xml') === \strlen($this->formsource) - 4) {
                    $this->formsource = Path::clean(JPATH_ROOT . '/' . $this->formsource);
                }

                break;
            case 'groupByFieldset':
                if ($value !== null) {
                    $value = (string)$value;
                    $this->groupByFieldset = !($value === 'false' || $value === 'off' || $value === '0');
                }
                break;

            case 'layout':
                $this->layout = (string)$value;

                // Make sure the layout is not empty.
                if (!$this->layout) {
                    $this->layout = 'modules.msg_rmf.form.field.sortablefields';
                }

                break;

            case 'value':
                // We allow a json encoded string or an array
                if (is_string($value)) {
                    $value = json_decode($value, true);
                }

                $this->value = $value !== null ? (array)$value : null;

                break;

            default:
                parent::__set($name, $value);
        }
    }

    /**
     * Method to attach a Form object to the field.
     *
     * @param \SimpleXMLElement $element The SimpleXMLElement object representing the <field /> tag for the form field object.
     * @param mixed $value The form field value to validate.
     * @param string $group The field name group control value.
     *
     * @return  boolean  True on success.
     *
     * @since   1.0.0
     */
    public function setup(\SimpleXMLElement $element, $value, $group = null)
    {
        if (!parent::setup($element, $value, $group)) {
            return false;
        }

        foreach (['formsource', 'layout', 'groupByFieldset'] as $attributeName) {
            $this->__set($attributeName, $element[$attributeName]);
        }

        if ((string)$element['fieldname']) {
            $this->__set('fieldname', $element['fieldname']);
        }

        if ($this->value && \is_string($this->value)) {
            // Guess here is the JSON string from 'default' attribute
            $this->value = json_decode($this->value, true);
        }

        if (!$this->formsource && $element->form) {
            // Set the formsource parameter from the content of the node
            $this->formsource = $element->form->saveXML();
        }

        $this->fields = $this->getFields();

        return true;
    }

    /**
     * Method to get the field input markup.
     *
     * @return  string  The field input markup.
     *
     * @throws \Exception
     * @since   1.0.0
     */
    protected function getInput()
    {
        // Prepare data for renderer
        $data = $this->getLayoutData();
        $control = $this->name;

        try {
            $form = $this->loadSubForm();
            $forms = $this->loadSubFormData($form);
        } catch (\Exception $e) {
            return $e->getMessage();
        }

        $data['forms'] = $forms;
        $data['control'] = $control;
        $data['fieldname'] = $this->fieldname;
        $data['fieldId'] = $this->id;
        $data['fields'] = $this->fields;
        $data['groupByFieldset'] = $this->groupByFieldset;

        /**
         * For each rendering process of a subform element, we want to have a
         * separate unique subform id present to could distinguish the eventhandlers
         * regarding adding/moving/removing rows from nested subforms from their parents.
         */
        static $unique_subform_id = 0;
        $data['unique_subform_id'] = ('sf-' . ($unique_subform_id++));

        // Prepare renderer
        $renderer = $this->getRenderer($this->layout);

        // Allow to define some Layout options as attribute of the element
        if ($this->element['component']) {
            $renderer->setComponent((string)$this->element['component']);
        }

        if ($this->element['client']) {
            $renderer->setClient((string)$this->element['client']);
        }

        // Render
        $html = $renderer->render($data);

        // for allow to submit an empty value
        $html = '<input name="' . $this->name . '" type="hidden" value="">' . $html;

        return $html;
    }

    /**
     * Method to get the name used for the field input tag.
     *
     * @param string $fieldName The field element name.
     *
     * @return  string  The name to be used for the field input tag.
     *
     * @since   1.0.0
     */
    protected function getName($fieldName)
    {
        $name = '';

        // If there is a form control set for the attached form add it first.
        if ($this->formControl) {
            $name .= $this->formControl;
        }

        // If the field is in a group add the group control to the field name.
        if ($this->group) {
            // If we already have a name segment add the group control as another level.
            $groups = explode('.', $this->group);

            if ($name) {
                foreach ($groups as $group) {
                    $name .= '[' . $group . ']';
                }
            } else {
                $name .= array_shift($groups);

                foreach ($groups as $group) {
                    $name .= '[' . $group . ']';
                }
            }
        }

        // If we already have a name segment add the field name as another level.
        if ($name) {
            $name .= '[' . $fieldName . ']';
        } else {
            $name .= $fieldName;
        }

        return $name;
    }

    /**
     * Loads the form instance for the subform.
     *
     * @return  Form  The form instance.
     *
     * @throws  \InvalidArgumentException if no form provided.
     * @throws  \RuntimeException if the form could not be loaded.
     *
     * @since   1.0.0
     */
    public function loadSubForm()
    {
        $control = $this->name;
        $control .= '[' . $this->fieldname . 'X]';

        // Prepare the form template
        $formname = 'subform.' . str_replace(['jform[', '[', ']'], ['', '.', ''], $this->name);
        $subForm = Factory::getContainer()->get(FormFactoryInterface::class)->createForm($formname, ['control' => $control]);
        $subForm->load($this->formsource);

        return $subForm;
    }

    /**
     * Binds given data to the subform and its elements.
     *
     * @param Form $subForm Form instance of the subform.
     *
     * @return  Form[]  Array of Form instances for the rows.
     *
     * @since   1.0.0
     */
    protected function loadSubFormData(Form $subForm)
    {
        $value = $this->value ? (array)$this->value : [];
        $fields = $this->fields;

        // Multiple rows possible: Construct array and bind values to their respective forms.
        $forms = [];

        foreach ($fields as $i => $item) {
            $control = $this->name . '[' . $item->name . ']';
            $itemForm = Factory::getContainer()->get(FormFactoryInterface::class)->createForm($item->name, ['control' => $control]);
            $itemForm->load($this->formsource);

            if (!empty($value[$item->name])) {
                $itemForm->bind($value[$item->name]);
            }

            $forms[$item->name] = $itemForm;
        }

        return $forms;
    }

    /**
     * Method to filter a field value.
     *
     * @param mixed $value The optional value to use as the default for the field.
     * @param string $group The optional dot-separated form group path on which to find the field.
     * @param Registry $input An optional Registry object with the entire data set to filter
     *                            against the entire form.
     *
     * @return  mixed   The filtered value.
     *
     * @throws  \UnexpectedValueException
     * @since   1.0.0
     */
    public function filter($value, $group = null, Registry $input = null)
    {
        // Make sure there is a valid SimpleXMLElement.
        if (!($this->element instanceof \SimpleXMLElement)) {
            throw new \UnexpectedValueException(sprintf('%s::filter `element` is not an instance of SimpleXMLElement', \get_class($this)));
        }

        // Get the field filter type.
        $filter = (string)$this->element['filter'];

        if ($filter !== '') {
            return parent::filter($value, $group, $input);
        }

        // Dirty way of ensuring required fields in subforms are submitted and filtered the way other fields are
        $subForm = $this->loadSubForm();

        // Subform field may have a default value, that is a JSON string
        if ($value && is_string($value)) {
            $value = json_decode($value, true);

            // The string is invalid json
            if (!$value) {
                return null;
            }
        }

        $return = [];

        if ($value) {
            $i = 1;
            foreach ($value as $key => $val) {
                $subFormVal = $subForm->filter($val);
                $subFormVal['ordering'] = $i++;
                $return[$key] = $subFormVal;
            }
        }

        return $return;
    }

    /**
     * Method to get the field options.
     *
     * @return  array  The field option objects.
     *
     * @throws  \Exception
     *
     * @since  1.0.0
     */
    protected function getFields(): array
    {
        $value = $this->value ? (array)$this->value : [];
        $items = FilterHelper::getAllFields();

        $params = ParamsHelper::getComponentParams();

        // Prepare options
        $fields = array();
        $defaultOrdering = count($items) + 5;

        foreach ($this->basicFields as $basicField) {
            // Add self made filter fields
            $field = new \stdClass();
            $field->name = $basicField['name'];
            $field->title = Text::_($basicField['title']);
            $field->group = 'Filter';
            $field->fieldset = '';
            $field->fieldset_name = '';
            $field->type = $basicField['type'];
            $field->ordering = isset($value[$basicField['name']]) && $value[$basicField['name']]['ordering'] ? $value[$basicField['name']]['ordering'] : $defaultOrdering;

            switch ($basicField['type'])
            {
                case "InStock":
                    $field->state = (bool)$params->get('stock_accounting', 0);
                    break;
                case "Price":
                    $field->state = !$params->get('hide_prices', 0);
                    break;
                default:
                    $field->state = null;
                    break;
            }

            $fields[] = $field;
        }

        foreach ($items as $item) {
            if ($item->area !== 'products'
                || (int)$item->params->get('display_filter', 0) !== 1) continue;

            $field = new \stdClass();
            $field->name = $item->alias;
            $field->title = $item->title;
            $field->group = 'ProductField';
            $field->fieldset = $item->fieldset_alias;
            $field->fieldset_name = $item->fieldset_name;

            $type = $item->params->get('type', 'unknown');

            if (isset($this->correctType[$item->plugin]))
            {
                if (is_array($this->correctType[$item->plugin]))
                {
                    $type = $this->correctType[$item->plugin][$type];
                } else {
                    $type = $this->correctType[$item->plugin];
                }
            } else {
                $type = $item->params->get('display_filter_as', $item->params->get('type', 'unknown'));
            }

            $field->type = $type;

            $field->ordering = isset($value[$item->alias]) && $value[$item->alias]['ordering'] ? $value[$item->alias]['ordering'] : $defaultOrdering;
            $field->state = $item->state && $item->params->get('display_filter', 1);

            $fields[] = $field;
        }

        usort($fields, function ($a, $b) {
            return $a->ordering <=> $b->ordering;
        });

        return $fields;
    }
}