<?php
/*
 * @package     mod_msg_rmf
 * @version     1.0.4
 * @author      Vladislav Tsygankov
 * @copyright   Copyright (c) 2023 MSGroup. All rights reserved.
 * @license     GNU/GPL license: https://www.gnu.org/copyleft/gpl.html
 * @link        https://msgru.com/
 */

namespace MSGJoomla\Module\RMFilter\Site\Helper;

// phpcs:disable PSR1.Files.SideEffects
defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

use Exception;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Form\FormFactoryInterface;
use Joomla\CMS\Form\FormField;
use Joomla\CMS\Language\Multilanguage;
use Joomla\CMS\MVC\Model\ModelInterface;
use Joomla\CMS\Router\Route;
use Joomla\Database\DatabaseInterface;
use Joomla\Registry\Registry;
use Joomla\Component\RadicalMart\Administrator\Helper\ParamsHelper;
use Joomla\Component\RadicalMart\Site\Helper\RouteHelper;
use Joomla\Component\RadicalMart\Site\Model\ProductsModel;
use Joomla\Component\RadicalMart\Administrator\Helper\FieldsHelper;
use Joomla\Utilities\ArrayHelper;
use MSGJoomla\Module\RMFilter\Site\Adapter\AdapterFactory;
use MSGJoomla\Module\RMFilter\Site\Driver\DriverFactory;

/**
 * Helper for mod_msg_rmf
 *
 * @since  1.0.0
 */
class FilterHelper
{
    protected static string $context = 'com_radicalmart.category';
    /**
     * Category id.
     *
     * @var  array|null
     *
     * @since  1.0.0
     */
    protected static ?array $_category = null;

    /**
     * Products filter form object.
     *
     * @var  array|null
     *
     * @since  1.0.0
     */
    protected static ?array $_form = null;

    /**
     * Category field groups.
     *
     * @var  array|null
     *
     * @since  1.0.0
     */
    protected static ?array $_categoryFieldSets = null;

    /**
     * ModelInterface objects.
     *
     * @var  ModelInterface[]|null
     *
     * @since  1.0.0
     */
    protected static ?array $_models = null;

    /**
     * @var array
     *
     * @since  1.0.0
     */
    protected static ?array $_categories = null;

    /**
     * Method to get filter form.
     *
     * @param Registry $params Module params.
     *
     * @return  Form|false  The Form object or false on error.
     *
     * @throws  Exception
     *
     * @since  1.0.0
     */

    public static function getForm(Registry $params)
    {
        /** @var Form $form */
        $form = Factory::getContainer()->get(FormFactoryInterface::class)->createForm('filter_products', ['load_data' => true]);

        $form->loadFile(JPATH_ROOT . '/modules/mod_msg_rmf/forms/msg_rmf_filter.xml', false);

        $data = Factory::getApplication()->getUserState(self::$context, new \stdClass());

        $categoryId = self::getCategoryId($params->get('category_id'));
        $categoryFields = FieldsHelper::getCategoryFields($categoryId);

        $adapterFactory = new AdapterFactory($categoryId, $categoryFields, $params, $data);

        foreach ($params->get('fields') as $filterField)
        {
            if(empty($filterField->show)) continue;

            $adapter = $adapterFactory->createAdapter($filterField->name, $filterField->custom_title, $filterField->group, $filterField->type);

            $adapter->handle();

            if ($adapter->isShow())
            {
                $fieldXml = $adapter->getXml();

                $form->setField($fieldXml, $adapter->getGroup(), true, $adapter->getFieldSet());
            }
        }

        $form->bind($data);

        return $form;
    }

    public static function getUserStateFromRequest($key, $request, $default = null, $type = 'none', $resetPage = true)
    {
        $app       = Factory::getApplication();
        $input     = $app->input;
        $old_state = $app->getUserState($key);
        $cur_state = $old_state ?? $default;
        $new_state = $input->get($request, null, $type);

        // BC for Search Tools which uses different naming
        if ($new_state === null && strpos($request, 'filter_') === 0) {
            $name    = substr($request, 7);
            $filters = $app->input->get('filter', [], 'array');

            if (isset($filters[$name])) {
                $new_state = $filters[$name];
            }
        }

        if ($cur_state != $new_state && $new_state !== null && $resetPage) {
            $input->set('limitstart', 0);
        }

        // Save the new value only if it is set in this request.
        if ($new_state !== null) {
            $app->setUserState($key, $new_state);
        } else {
            $new_state = $cur_state;
        }

        return $new_state;
    }

    /**
     * Method to get correct category id.
     *
     * @param int|null $pk Category id.
     *
     * @return  int  Correct category id.
     *
     * @throws  Exception
     *
     * @since  1.0.0
     */
    public static function getCategoryId(int $pk = null): int
    {
        if (self::$_category === null) {
            self::$_category = [];
        }

        if (!isset(self::$_category[$pk])) {
            $category = 1;
            $app = Factory::getApplication();

            if ((int)$pk) {
                $category = $pk;
            } elseif ($app->input->get('option') === 'com_radicalmart'
                && $app->input->get('view') === 'category'
                && $app->input->getInt('id')) {
                $category = $app->input->getInt('id');
            }

            self::$_category[$pk] = $category;

            return $category;
        }

        return self::$_category[$pk];
    }

    /**
     * Method to get products categories.
     *
     * @param   string|array  $pks  The id of the categories.
     *
     * @throws  \Exception
     *
     * @return  object[] Categories array.
     *
     * @since  1.0.0
     */
    public static function getCategories($pks = null): array
    {
        if (self::$_categories === null)
        {
            self::$_categories = [];
        }
        if (!is_array($pks))
        {
            $pks = array_unique(ArrayHelper::toInteger(explode(',', $pks)));
        }
        if (empty($pks))
        {
            return array();
        }

        // Check loaded categories
        $categories = [];
        $get = [];
        foreach ($pks as $pk)
        {
            if (isset(self::$_categories[$pk]))
            {
                $categories[$pk] = self::$_categories[$pk];
            }
            else $get[] = $pk;
        }

        // Get categories
        if (!empty($get))
        {
            /* @var ProductModel $model */
            $model = self::getModel('Product', 'Site');
            $model->setState('filter.published', 1);

            if ($rows = $model->getCategories($get))
            {
                foreach ($rows as $row)
                {
                    self::$_categories[$row->id] = $row;
                    $categories[$row->id]        = $row;
                }
            }
        }

        return $categories;
    }

    /**
     * Method to get products model.
     *
     * @return  ModelInterface  Products mode.
     *
     * @throws  Exception
     *
     * @since  1.0.0
     */
    protected static function getModel(string $modelName, string $applicationName = 'Site'): ModelInterface
    {
        if (!isset(self::$_models[$modelName])) {
            $app = Factory::getApplication();

            // Load language
            $app->getLanguage()->load('com_radicalmart');

            // Get model
            $model = $app->bootComponent('com_radicalmart')->getMVCFactory()
                ->createModel($modelName, $applicationName, ['ignore_request' => true]);
            $model->set('context', self::$context);
            $model->setState('params', ParamsHelper::getComponentParams());
            $model->setState('filter.language', Multilanguage::isEnabled());

            self::$_models[$modelName] = $model;
        }

        return self::$_models[$modelName];
    }

    /**
     * Method to get form action url.
     *
     * @param Registry $params Module params.
     *
     * @return  string  The action url.
     *
     * @throws  Exception
     *
     * @since  1.0.0
     */
    public static function getAction(Registry $params): string
    {
        $pk = self::getCategoryId($params->get('category'));

        return Route::link('site', RouteHelper::getCategoryViewRoute($pk));
    }

    public static function getAllFields(): array
    {
        $fields = [];

        $fieldsModel = self::getModel('Fields', 'Administrator');
        $items = $fieldsModel->getItems();
        $fieldsets = self::getFieldsets(ArrayHelper::getColumn($items, 'fieldset'));

        foreach ($items as $item) {
            $item->fieldset_alias = (isset($fieldsets[$item->fieldset])) ? $fieldsets[$item->fieldset]->alias : '';

            $fields[$item->alias] = $item;
        }

        return $fields;
    }

    /**
     * Method to get category fields.
     *
     * @param   int|null  $pk  The category id.
     *
     * @throws  \Exception
     *
     * @return object[] Category fields data.
     *
     * @since  1.0.0
     */
    public static function getCategoryFields(int $pk = null, string $column = 'id'): array
    {
        $fields = array();

        if(!in_array($column, array('id', 'alias'))) return $fields;

        foreach (FieldsHelper::getCategoryFields($pk) as $field)
        {
            $fields[$field->$column] = $field;
        }

        return $fields;
    }

    /**
     * Method to get fieldsets.
     *
     * @param   string|int[]  $pks  The name of the fieldsets.
     *
     * @return  object[] Fieldsets array.
     *
     * @since  1.0.0
     */
    protected static function getFieldsets($pks = null): array
    {
        // Prepare ids
        $fieldsets = [];
        if (!is_array($pks))
        {
            $pks = array_unique(explode(',', $pks));
        }
        if (empty($pks))
        {
            return $fieldsets;
        }

        // Check loaded fieldsets
        $get = [];
        foreach ($pks as $pk)
        {
            if (isset(self::$_categoryFieldSets[$pk]))
            {
                $fieldsets[$pk] = self::$_categoryFieldSets[$pk];
            }
            else $get[] = $pk;
        }

        // Get fieldsets
        if (!empty($get))
        {
            $db = Factory::getContainer()->get(DatabaseInterface::class);
            $query = $db->getQuery(true)
                ->select(['fs.id', 'fs.title', 'fs.alias'])
                ->from($db->quoteName('#__radicalmart_fieldsets', 'fs'))
                ->whereIn('fs.id', $get);
            if ($rows = $db->setQuery($query)->loadObjectList())
            {
                foreach ($rows as $row)
                {
                    self::$_categoryFieldSets[$row->id] = $row;
                    $fieldsets[$row->id]        = $row;
                }
            }
        }

        return $fieldsets;
    }
}
