<?php
/*
 * @package     mod_msg_rmf
 * @version     1.0.5
 * @author      Vladislav Tsygankov
 * @copyright   Copyright (c) 2023 MSGroup. All rights reserved.
 * @license     GNU/GPL license: https://www.gnu.org/copyleft/gpl.html
 * @link        https://msgru.com/
 */

namespace MSGJoomla\Module\RMFilter\Site\Adapter\ProductField;

// no direct access
\defined('JPATH_PLATFORM') or die;

use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\CMS\Cache\Controller\CallbackController;
use Joomla\CMS\Cache\Exception\CacheExceptionInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\FormField;
use Joomla\CMS\Language\Multilanguage;
use Joomla\Database\DatabaseInterface;
use Joomla\Database\ParameterType;
use Joomla\Component\RadicalMart\Site\Mapping\CategoryMapping;
use Joomla\Component\RadicalMart\Administrator\Helper\PriceHelper;
use Joomla\Utilities\ArrayHelper;
use MSGJoomla\Module\RMFilter\Site\Helper\FilterHelper;

/**
 * Adapter for joomla field type list
 *
 * @since  1.0.0
 */
class ListFieldAdapter extends BaseFieldAdapter
{
    protected string $layout = 'modules.msg_rmf.filter.field.list';

    protected string $type = 'list';
    protected $field;
    protected bool $multiSelect = false;

    /**
     * @throws \Exception
     */
    public function setup(): void
    {
        parent::setup();

        $this->multiSelect = $this->field->params->get('type', 'unknown') == 'checkboxes' || $this->field->params->get('multiple', false);
    }

    public function handle(): void
    {
        parent::handle();
        if (!$this->show) return;

        $caching = !Factory::getApplication()->get('debug');

        /** @var CallbackController $cache */
        $cache = Factory::getContainer()->get(CacheControllerFactoryInterface::class)->createCacheController('callback', ['defaultgroup' => 'mod_msg_rmf', 'caching' => $caching]);

        $cache->setLifeTime($this->factory->moduleParams->get('filter_cache_time', Factory::getApplication()->get('cachetime') * 60) / 60);

        $show_empty_options = $this->factory->moduleParams->get('show_empty_options', 'show');

        if (in_array($show_empty_options, array('hide', 'disable')))
        {
            try {
                $disabledOptions = $cache->get([$this, 'setOptions'], array(), md5(serialize(array($this->factory->categoryId, $this->field->id, $this->multiSelect))));
            } catch (CacheExceptionInterface $e) {
                $disabledOptions = $this->setOptions();
            }

            $needShow = false;

            foreach ($this->xml->children() as $option)
            {
                $value = (string)$option->attributes()->value;

                $disabled = !empty($value) && in_array($value, $disabledOptions);

                $option->addAttribute('disabled', $disabled);

                if (!$disabled) {
                    $needShow = true;
                }
            }

            $this->setAttribute('data-hide-options', $show_empty_options == 'hide');

            $this->show = $needShow;
        }
    }

    /**
     * Method to get field options with hidden or disable attr.
     *
     * @param   FormField  $field  Form field.
     * @param   int  $category_id  RadicalMart category id.
     * @param   bool  $show  set disable or hidden attribute. If true, that use disable.
     *
     * @throws  \Exception
     *
     * @return  array  Field options.
     *
     * @since  1.0.0
     */

    public function setOptions(): array
    {
        /** @var DatabaseInterface $db */
        $db = Factory::getContainer()->get(DatabaseInterface::class);
        $app      = Factory::getApplication();

        // Set published && debug state
        if ($app->input->getInt('debug', 0))
        {
            $published = [0, 1];
        }
        else
        {
            $published = 1;
        }

        $query = $db->getQuery(true)
            ->from($db->quoteName('#__radicalmart_products', 'p'));

        if($this->multiSelect) {
            $query->select('DISTINCT ' . $db->quoteName('json_table.option_value'))
                ->from('JSON_TABLE('.$db->quoteName('p.fields').', ' . $db->quote('$."' . $this->field->alias . '"[*]') . ' COLUMNS (option_value VARCHAR(255) PATH "$")) ' . $db->quoteName('json_table'));
        } else {
            $query->select('DISTINCT JSON_UNQUOTE(JSON_EXTRACT('.$db->quoteName('p.fields').', '.$db->quote('$."' . $this->field->alias . '"') .')) AS ' .  $db->quoteName('options'));
        }

        // Filter by language state
        if (Multilanguage::isEnabled())
        {
            $query->whereIn('p.language', [Factory::getApplication()->getLanguage()->getTag(), '*'],
                ParameterType::STRING);
        }

        if (is_numeric($published))
        {
            $query->where($db->quoteName('p.state') . ' = :published');

            $query->bind(':published', $published, ParameterType::INTEGER);
        }
        elseif (is_array($published))
        {
            $query->whereIn($db->quoteName('p.state'), $published);
        }

        // Filter by category state
        $category = $this->factory->categoryId;
        $conditionsCategory = ['FIND_IN_SET(' . $category . ', p.categories)'];
        foreach (CategoryMapping::getSubCategories($category) as $catid)
        {
            $conditionsCategory[] = 'FIND_IN_SET(' . $catid . ', p.categories)';
        }
        $query->extendWhere('AND', $conditionsCategory, 'OR');

        try {
            $actualOptions = $db->setQuery($query)->loadColumn();
        } catch (\Exception $e) {
            $actualOptions = array();
        }
        $disabled = array();

        foreach ($this->xml->children() as $option)
        {
            $value = (string)$option->attributes()->value;

            if (!in_array($value, $actualOptions)) {
                $disabled[] = $value;
            }
        }

        return $disabled;
    }
}