<?php
/*
 * @package     mod_msg_rmf
 * @version     __DEPLOY_VERSION__
 * @author      Vladislav Tsygankov
 * @copyright   Copyright (c) 2023 MSGroup. All rights reserved.
 * @license     GNU/GPL license: https://www.gnu.org/copyleft/gpl.html
 * @link        https://msgru.com/
 */

namespace MSGJoomla\Module\RMFilter\Site\Adapter\ProductField;

// no direct access
\defined('JPATH_PLATFORM') or die;

use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\CMS\Cache\Controller\CallbackController;
use Joomla\CMS\Cache\Exception\CacheExceptionInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\FormField;
use Joomla\CMS\Language\Multilanguage;
use Joomla\Database\DatabaseInterface;
use Joomla\Database\Exception\ExecutionFailureException;
use Joomla\Database\ParameterType;
use Joomla\Component\RadicalMart\Site\Mapping\CategoryMapping;
use Joomla\Component\RadicalMart\Administrator\Helper\PriceHelper;
use Joomla\Utilities\ArrayHelper;
use MSGJoomla\Module\RMFilter\Site\Helper\FilterHelper;

/**
 * Adapter for joomla field type list
 *
 * @since  1.0.0
 */
class ListFieldAdapter extends BaseFieldAdapter
{
    protected string $layout = 'modules.msg_rmf.filter.field.list';

    protected string $type = 'list';
    protected $field;
    protected bool $multiSelect = false;

    /**
     * @throws \Exception
     */
    public function setup(): void
    {
        parent::setup();

        $this->multiSelect = $this->field->params->get('type', 'unknown') == 'checkboxes' || $this->field->params->get('multiple', false);
    }

    public function handle(): void
    {
        parent::handle();
        if (!$this->show) return;

        /** @var CallbackController $cache */
        $cache = Factory::getContainer()->get(CacheControllerFactoryInterface::class)->createCacheController('callback', ['defaultgroup' => 'mod_msg_rmf', 'caching' => true, 'application' => 'site', 'language' => $this->factory->params->get('language', 'en-GB')]);

        $cacheId = md5(serialize(array($this->factory->categoryId, $this->field->id, $this->multiSelect)));

        $cache->cache->setLifeTime($this->factory->params->get('filter_cache_time', Factory::getApplication()->get('cachetime')));

        $show_empty_options = $this->factory->params->get('show_empty_options', 'show');
        try {
            $disabledOptions = $cache->get([$this, 'setOptions'], array(), $cacheId);
        } catch (CacheExceptionInterface $cacheException) {
            try {
                $disabledOptions = $this->setOptions();
            } catch (ExecutionFailureException $databaseException) {
                $disabledOptions = [];
            }
        } catch (ExecutionFailureException $databaseException) {
            $disabledOptions = [];
        }

        if (in_array($show_empty_options, array('hide', 'disable')))
        {
            $needShow = false;

            foreach ($this->xml->children() as $option)
            {
                $value = (string)$option->attributes()->value;

                $disabled = !empty($value) && in_array($value, $disabledOptions);

                $option->addAttribute('disabled', $disabled);

                if (!$disabled) {
                    $needShow = true;
                }
            }

            $this->setAttribute('data-hide-options', $show_empty_options == 'hide');

            $this->show = $needShow;
        }
    }

    /**
     * Method to get field options with hidden or disable attr.
     *
     * @return  array  Field options.
     *
     * @throws \Exception
     * @since  1.0.0
     */

    public function setOptions(): array
    {
        /** @var DatabaseInterface $db */
        $db = Factory::getContainer()->get(DatabaseInterface::class);
        $app      = Factory::getApplication();

        $published = 1;

        $query = $db->getQuery(true)
            ->from($db->quoteName('#__radicalmart_products', 'p'));

        if($this->multiSelect) {
            $query->select('DISTINCT ' . $db->quoteName('json_table.option_value'))
                ->from('JSON_TABLE('.$db->quoteName('p.fields').', ' . $db->quote('$."' . $this->field->alias . '"[*]') . ' COLUMNS (option_value VARCHAR(255) PATH "$")) ' . $db->quoteName('json_table'));
        } else {
            $query->select('DISTINCT JSON_UNQUOTE(JSON_EXTRACT('.$db->quoteName('p.fields').', '.$db->quote('$."' . $this->field->alias . '"') .')) AS ' .  $db->quoteName('options'));
        }

        // Filter by language state
        if (Multilanguage::isEnabled())
        {
            $query->whereIn('p.language', [Factory::getApplication()->getLanguage()->getTag(), '*'],
                ParameterType::STRING);
        }

        if (is_numeric($published))
        {
            $query->where($db->quoteName('p.state') . ' = :published');

            $query->bind(':published', $published, ParameterType::INTEGER);
        }
        elseif (is_array($published))
        {
            $query->whereIn($db->quoteName('p.state'), $published);
        }

        // Filter by category state
        $category_id = $this->factory->categoryId;
        $conditionsCategory = ['FIND_IN_SET(' . $category_id . ', p.categories)'];
        foreach (CategoryMapping::getSubCategories($category_id) as $catid)
        {
            $conditionsCategory[] = 'FIND_IN_SET(' . $catid . ', p.categories)';
        }
        $query->extendWhere('AND', $conditionsCategory, 'OR');

        try {
            $actualOptions = $db->setQuery($query)->loadColumn();
        } catch (\Exception $e) {
            $actualOptions = array();
        }
        $disabled = array();

        foreach ($this->xml->children() as $option)
        {
            $value = (string)$option->attributes()->value;

            if (!in_array($value, $actualOptions)) {
                $disabled[] = $value;
            }
        }

        return $disabled;
    }
}
