<?php
/**
 * @copyright   (c) 2013-2025 Nekrasov Vitaliy <nekrasov_vitaliy@list.ru>
 * @license     GNU General Public License version 2 or later
 */
namespace Joomla\Plugin\JShopping\WishboxQuickOrder\Administrator\Model;

use Exception;
use Joomla\CMS\Factory;
use Joomla\Component\Jshopping\Site\Helper\Request;
use Joomla\Component\Jshopping\Site\Model\CheckoutModel;
use Joomla\Component\Jshopping\Site\Model\CheckoutOrderModel;
use Joomla\Component\Jshopping\Site\Model\CheckoutPaymentModel;
use Joomla\Component\Jshopping\Site\Model\CheckoutShippingModel;
use Joomla\Component\Jshopping\Site\Table\ConfigTable;
use Joomla\Component\Jshopping\Site\Table\ProductTable;
use Joomla\Component\Jshopping\Site\Table\ShippingMethodTable;
use Joomla\Event\Event;
use Joomla\Registry\Registry;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\User\User;
use Joomla\Component\Jshopping\Site\Lib\JSFactory;
use Joomla\Component\Jshopping\Site\Lib\MultiLangfield;
use Joomla\Component\Jshopping\Site\Model\CartModel;
use Joomla\Component\Jshopping\Site\Model\UserGuestModel;
use Joomla\Component\Jshopping\Site\Table\UserShopTable;
use UnexpectedValueException;
use function get_class;
use function sprintf;

// phpcs:disable PSR1.Files.SideEffects
defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * @since 1.0.0
 */
class OrdersModel extends BaseDatabaseModel
{
	/**
	 * The URL option for the component.
	 *
	 * @var    string
	 *
	 * @since  1.0.0
	 */
	protected $option = '';

	/**
	 * @var User
	 *
	 * @since 1.0.0
	 */
	protected User $user;

	/**
	 * @var CartModel
	 *
	 * @since 1.0.0
	 */
	protected CartModel $cart;

	/**
	 * @var Multilangfield
	 *
	 * @since 1.0.0
	 */
	protected Multilangfield $lang;

	/**
	 * @var UserguestModel|UsershopTable
	 *
	 * @since 1.0.0
	 */
	protected UserguestModel|UsershopTable $advUser;

	/**
	 * Constructor
	 *
	 * @param   array                     $config   Config
	 * @param   MVCFactoryInterface|null  $factory  Factory
	 *
	 * @throws Exception
	 *
	 * @since 1.0.0
	 */
	public function __construct($config = [], MVCFactoryInterface $factory = null)
	{
		parent::__construct($config, $factory);

		$app = Factory::getApplication();
		$this->advUser = JSFactory::getUser()->loadDataFromEdit();
		$this->user = $app->getIdentity();
		$this->cart = JSFactory::getModel('cart', 'Site');
		$this->cart->load();
		$this->lang = JSFactory::getLang();
		$plugin = PluginHelper::getPlugin('jshopping', 'wishboxquickorder');
		$this->pluginParams = new Registry($plugin->params);
	}

	/**
	 * @param   integer  $productId           Product id
	 * @param   float    $quantity            Quantity
	 * @param   string   $paymentMethodClass  Payment method class
	 *
	 * @return   boolean
	 *
	 * @throws Exception
	 *
	 * @since 1.0.0
	 */
	public function create(int $productId, float $quantity, string $paymentMethodClass): bool
	{
		$app = Factory::getApplication();
		$input = $app->getInput();

		if (!$productId)
		{
			throw new Exception('Product id must be more than zero', 500);
		}

		$attribut = Request::getAttribute();
		$freeattribut = Request::getFreeAttribute();

		$this->addProductToCart($productId, $quantity, $attribut, $freeattribut);

		$post = [
			'f_name'    => $input->post->getString('f_name', ''),
			'l_name'    => $input->post->getString('l_name', ''),
			'email'     => $input->post->getString('email', ''),
			'phone'     => $input->post->getString('phone', ''),
			'street'    => $input->post->getString('street', ''),
		];

		$this->step2($post);

		$this->step3($paymentMethodClass);
		$this->step4();
		$this->step5();

		return true;
	}

	/**
	 * @param   integer  $productId     Product id
	 * @param   float    $quantity      Quantity
	 * @param   array    $attribut      Attributes
	 * @param   array    $freeattribut  Free attributes
	 *
	 * @return void
	 *
	 * @throws Exception
	 * @since 1.0.0
	 */
	private function addProductToCart(int $productId, float $quantity, array $attribut, array $freeattribut): void
	{
		$app = Factory::getApplication();
		$this->cart->deleteAll();

		if (!$productId)
		{
			throw new Exception('Product id must be more than zero', 500);
		}

		$nameField = $this->lang->get('name');
		$addToCartErrorFlag = false;

		/** @var ProductTable $productTable */
		$productTable = JSFactory::getTable('product');

		if (!$productTable->load($productId))
		{
			$app->enqueueMessage('Product with id ' . $productId . ' not found', 'warning');
		}

		$errors = [];

		if (!$this->cart->add($productId, $quantity, $attribut, $freeattribut, [], 1, $errors, 0))
		{
			$message = $productTable->$nameField;
			$message .= ':';

			foreach ($errors as $error)
			{
				$message .= ' '.$error;
			}

			$app->enqueueMessage($message, 'warning');

			$addToCartErrorFlag = true;
		}

		if ($addToCartErrorFlag)
		{
			throw new Exception(null, 500);
		}

		if ($this->cart->getCountProduct() == 0)
		{
			throw new Exception('Cart is empty', 500);
		}
	}

	/**
	 * @param   array  $post  Post
	 *
	 * @return void
	 *
	 * @throws Exception
	 *
	 * @since 1.0.0
	 *
	 * @noinspection PhpUnusedParameterInspection
	 */
	private function step2(array $post = []): void
	{
		$this->triggerEvent(
			'onBeforeSaveCheckoutStep2',
			[
				&$this->advUser,
				&$this->user,
				&$this->cart,
				&$usereditModel
			]
		);


		$advUser = JSFactory::getUser();
		$advUser->bind($post);
		$advUser->store();

		/** @var CheckoutModel $checkoutModel */
		$checkoutModel = JSFactory::getModel('checkout', 'Site');

		if ($this->getState('enable_check_step'))
		{
			$checkoutModel->checkStep(2);
		}

		$checkoutModel->setCart($this->cart);
		$checkoutModel->setEmptyCheckoutPrices();

		$this->triggerEvent('onAfterSaveCheckoutStep2', [&$this->advUser, &$this->user, &$this->cart]);

		$checkoutModel->setMaxStep(3);
	}

	/**
	 * @param   string  $paymentMethodClass  Payment method class
	 *
	 * @return void
	 *
	 * @throws Exception
	 *
	 * @since 1.0.0
	 */
	private function step3(string $paymentMethodClass): void
	{
		$post = [
			'payment_method' => $paymentMethodClass
		];

		/** @var CheckoutPaymentModel $checkoutpaymentModel */
		$checkoutpaymentModel = JSFactory::getModel('checkoutPayment', 'Site');

		if ($this->getState('enable_check_step'))
		{
			$checkoutpaymentModel->checkStep(3);
		}

		$checkoutpaymentModel->setCart($this->cart);
		$this->triggerEvent('onBeforeSaveCheckoutStep3save', [&$post]);
		$params = [];

		if (!$checkoutpaymentModel->savePaymentData($paymentMethodClass, $params, $this->advUser))
		{
			throw new Exception(__FILE__ . ':' . __LINE__ . ' ' . $checkoutpaymentModel->getError(), 500);
		}

		$paymentMethod = $checkoutpaymentModel->getActivePaymMethod();
		$this->triggerEvent('onAfterSaveCheckoutStep3save', [&$advUser, &$paymentMethod, &$this->cart]);
		$checkoutpaymentModel->setMaxStep(4);
	}

	/**
	 * @since 1.0.0
	 */
	private function step4(): void
	{
		/** @var ConfigTable $jshopConfig */
		$jshopConfig = JSFactory::getConfig();

		/** @var CheckoutShippingModel $checkoutshippingModel */
		$checkoutshippingModel = JSFactory::getModel('checkoutShipping', 'Site');

		if ($this->getState('enable_check_step'))
		{
			$checkoutshippingModel->checkStep(4);
		}

		$this->triggerEvent('onBeforeSaveCheckoutStep4save');

		$shippingMethodId = (int) $this->pluginParams->get('shipping_method_id');

		/** @var ShippingMethodTable $shippingMethodTable */
		$shippingMethodTable = JSFactory::getTable('shippingMethod');

		$shippingMethodTable->load($shippingMethodId);

		/** @noinspection PhpUndefinedFieldInspection */
		$shippingPriceId = $shippingMethodTable->getShippingPriceId($shippingMethodId, $jshopConfig->default_country);

		$allparams = [];
		$checkoutshippingModel->setCart($this->cart);

		if (!$checkoutshippingModel->saveShippingData($shippingPriceId, $allparams, $this->advUser))
		{
			throw new Exception($checkoutshippingModel->getError(), 500);
		}

		$shMethod = $checkoutshippingModel->getActiveShippingMethod();
		$shippingMethodPrice = $checkoutshippingModel->getActiveShippingMethodPrice();
		$this->triggerEvent('onAfterSaveCheckoutStep4', [&$advUser, &$shMethod, &$shippingMethodPrice, &$this->cart]);

		$checkoutshippingModel->setMaxStep(5);
	}

	/**
	 * @since 1.0.0
	 */
	private function step5(): void
	{
		/** @var CheckoutOrderModel $checkoutOrderModel */
		$checkoutOrderModel = JSFactory::getModel('checkoutOrder', 'Site');

		if ($this->getState('enable_check_step'))
		{
			$checkoutOrderModel->checkStep(5);
		}

		$post = [];
		$this->triggerEvent('onLoadStep5save', [1]);
		$checkoutOrderModel->setCart($this->cart);

		if (!$this->cart->checkListProductsQtyInStore())
		{
			throw new Exception('', 500);
		}

		$orderTable = $checkoutOrderModel->orderDataSave($this->advUser, $post);
		$this->setState('order_id', $orderTable->order_id);
		$this->setState('order_number', $orderTable->order_number);
		$this->triggerEvent('onEndCheckoutStep5', [&$orderTable, &$this->cart]);
	}

	/**
	 * @throws Exception
	 *
	 * @since 1.0.0
	 */
	protected function triggerEvent(string $eventName, array $args = [])
	{
		$app = Factory::getApplication();

		try
		{
			$dispatcher = $this->getDispatcher();
		}
		catch (UnexpectedValueException)
		{
			$app->getLogger()->error(sprintf('Dispatcher not set in %s, cannot trigger events.', get_class($this)));

			return [];
		}

		$event = new Event($eventName, $args);

		$result = $dispatcher->dispatch($eventName, $event);

		return !isset($result['result']) ? [] : $result['result'];
	}
}
