<?php
/**
 * @copyright   (c) 2013-2025 Nekrasov Vitaliy <nekrasov_vitaliy@list.ru>
 * @license     GNU General Public License version 2 or later
 */
namespace Joomla\Plugin\JShopping\WishboxQuickOrder\Extension;

use Exception;
use Joomla\Application\ApplicationEvents;
use Joomla\CMS\Event\Application\AfterRenderEvent;
use Joomla\CMS\Event\Plugin\AjaxEvent;
use Joomla\CMS\Factory;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\LayoutHelper;
use Joomla\CMS\MVC\Factory\MVCFactoryServiceTrait;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Response\JsonResponse;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Session\Session;
use Joomla\Component\Jshopping\Site\Helper\Helper;
use Joomla\Component\Jshopping\Site\Lib\JSFactory;
use Joomla\Component\Jshopping\Site\Lib\MultiLangfield;
use Joomla\Component\Jshopping\Site\Table\ConfigTable;
use Joomla\Component\Jshopping\Site\Table\CurrencyTable;
use Joomla\Component\Jshopping\Site\Table\PaymentMethodTable;
use Joomla\Component\Jshopping\Site\Table\ProductTable;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Event\Dispatcher;
use Joomla\Event\Event;
use Joomla\Event\SubscriberInterface;
use Joomla\Plugin\JShopping\WishboxQuickOrder\Administrator\Model\OrdersModel;
use stdClass;
use function defined;

// phpcs:disable PSR1.Files.SideEffects
defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * @since 1.0.0
 */
final class WishboxQuickOrder extends CMSPlugin implements SubscriberInterface
{
	use MVCFactoryServiceTrait;
	use DatabaseAwareTrait;

	/**
	 * Load the language file on instantiation.
	 *
	 * @var    boolean
	 *
	 * @since  1.0.0
	 */
	protected $autoloadLanguage = true;

	/**
	 * Returns an array of events this subscriber will listen to.
	 *
	 * @return  array
	 *
	 * @since   1.0.0
	 */
	public static function getSubscribedEvents(): array
	{
		return [
			'onAjaxWishboxquickordersubmitform'         => 'onAjaxWishboxQuickOrderSubmitForm',
			'onAjaxWishboxquickordergetproductdata'     => 'onAjaxWishboxQuickOrderGetProductData',
			'onBeforeDisplayProductListView'            => 'onBeforeDisplayProductListView',
			'onBeforeDisplayProductView'                => 'onBeforeDisplayProductView',
			'onBeforeCheckUserEmailExistJshopUserShop'  => 'onBeforeCheckUserEmailExistJshopUserShop',
			'onAfterRender'                             => 'onAfterRender',
		];
	}

    /**
     * @param   Event  $event  Event
     *
     * @return void
     *
     * @since   1.0.0
     *
     * @noinspection PhpUnused
     * @noinspection PhpUndefinedFieldInspection
     */
    public function onBeforeDisplayProductListView(Event $event): void
    {
		$view = $event->getArgument(0);

	    if (!count($view->rows))
	    {
			return;
	    }

	    $db = $this->getDatabase();

		$app = $this->getApplication();
	    $document = $app->getDocument();

		/** @var ConfigTable $jshopConfig */
		$jshopConfig = JSFactory::getConfig();

	    /** @noinspection PhpUndefinedFieldInspection */
	    $defaultCurrencyId = $jshopConfig->mainCurrency;

		$style = $this->params->get('style', 'bootstrap');

		/** @var CurrencyTable $defaultCurrencyTable */
		$defaultCurrencyTable = JSFactory::getTable('currency');

		$defaultCurrencyTable->load($defaultCurrencyId);

		$scriptOptions = [
			'controller'        => Route::_('index.php?option=com_ajax&plugin=wishboxquickordergetform&group=jshopping&format=json'),
			'csrf'              => Session::getFormToken(),
			'currency_code'     => $defaultCurrencyTable->currency_code,
			'format_currency'   => $jshopConfig->format_currency[$jshopConfig->currency_format],
			'decimal_count'     => $jshopConfig->decimal_count,
			'decimal_symbol'    => $jshopConfig->decimal_symbol,
		];

	    $document->addScriptOptions('wishboxquickorder', $scriptOptions);

		/** @var stdClass[] $products */
	    $products = $view->rows;

		$productIds = [];

	    foreach ($products as $product)
	    {
			$productIds[] = $product->product_id;
	    }

	    $query = $db->createQuery()
		    ->select(
			    [
				    'product_id',
				    'attr_id'
			    ]
		    )
		    ->from($db->qn('#__jshopping_products_attr2'))
		    ->whereIn('product_id', $productIds)
		    ->group('product_id');
	    $db->setQuery($query);
	    $attrIds = $db->loadAssocList('product_id', 'attr_id');

		$productListPosition = $this->params->get('product_list_position', '_tmp_var_buttons');

		foreach ($products as $k => $product)
		{
			$buttonHtml = LayoutHelper::render(
				'plugins.jshopping.wishboxquickorder.' . $style . '.button',
				[
					'product'       => $product,
					'categoryId'    => $product->category_id,
					'enableModal'   => !isset($attrIds[$product->product_id]) || !$attrIds[$product->product_id],
				]
			);

			$products[$k]->$productListPosition .= $buttonHtml;
		}
    }

	/**
	 * @param   Event  $event  Event
	 *
	 * @return void
	 *
	 * @since   1.0.0
	 *
	 * @noinspection PhpUnused
	 * @noinspection PhpUndefinedFieldInspection
	 */
	public function onBeforeDisplayProductView(Event $event): void
	{
		$view = $event->getArgument(0);

		$app = $this->getApplication();
		$document = $app->getDocument();

		/** @var ConfigTable $jshopConfig */
		$jshopConfig = JSFactory::getConfig();

		/** @noinspection PhpUndefinedFieldInspection */
		$defaultCurrencyId = $jshopConfig->mainCurrency;

		$style = $this->params->get('style', 'bootstrap');

		/** @var CurrencyTable $defaultCurrencyTable */
		$defaultCurrencyTable = JSFactory::getTable('currency');

		$defaultCurrencyTable->load($defaultCurrencyId);

		$scriptOptions = [
			'controller'        => Route::_('index.php?option=com_ajax&plugin=wishboxquickordergetform&group=jshopping&format=json'),
			'csrf'              => Session::getFormToken(),
			'currency_code'     => $defaultCurrencyTable->currency_code,
			'format_currency'   => $jshopConfig->format_currency[$jshopConfig->currency_format],
			'decimal_count'     => $jshopConfig->decimal_count,
			'decimal_symbol'    => $jshopConfig->decimal_symbol,
		];

		$document->addScriptOptions('wishboxquickorder', $scriptOptions);

		HTMLHelper::_('behavior.formvalidator');

		$productPosition = $this->params->get('product_position', '_tmp_var_buttons');

		$buttonHtml = LayoutHelper::render(
			'plugins.jshopping.wishboxquickorder.' . $style . '.button',
			[
				'product'       => $view->product,
				'categoryId'    => $view->product->getCategory(),
				'enableModal'   => true,
			]
		);

		$view->$productPosition .= $buttonHtml;
	}

	/**
	 * @param   AjaxEvent  $event  Event
	 *
	 * @return void
	 *
	 * @since   1.0.0
	 *
	 * @noinspection PhpUnused
	 * @noinspection PhpUndefinedFieldInspection
	 */
	public function onAjaxWishboxQuickOrderGetProductData(AjaxEvent $event): void
	{
		require_once(JPATH_SITE . '/components/com_jshopping/bootstrap.php');
		require_once(JPATH_SITE . '/components/com_jshopping/loadparams.php');

		$app = $this->getApplication();

		/** @var Multilangfield $lang */
		$lang = JSFactory::getLang();

		$nameField = $lang->get('name');

		/** @var ConfigTable $jshopConfig */
		$jshopConfig = JSFactory::getConfig();

		$productId = $app->input->getInt('product_id', 0);

		/** @var ProductTable $productTable */
		$productTable = JSFactory::getTable('product');

		$productTable->load($productId);

		$productTable->wishboxfoldquantity = (float) $productTable->wishboxfoldquantity;

		if ($jshopConfig->use_decimal_qty) // phpcs:ignore
		{
			$productTable->wishboxfoldquantity = floatval(str_replace(',', '.', $productTable->wishboxfoldquantity));
			$productTable->wishboxfoldquantity = round($productTable->wishboxfoldquantity, $jshopConfig->cart_decimal_qty_precision); // phpcs:ignore
		}

		$result = [
			'productName'           => $productTable->$nameField,
			'productEan'            => $productTable->product_ean,
			'productPrice'          => $productTable->product_price,
			'productImage'          => Helper::getPatchProductImage($productTable->image, '', 1),
			'wishboxfoldquantity'   => $productTable->wishboxfoldquantity ?? 1,
			'stock'                 => $productTable->unlimited ? 'Unlimited' : Helper::formatqty($productTable->product_quantity),
		];

		$event->setArgument('result', $result);
	}

	/**
	 * @param   AjaxEvent  $event  Event
	 *
	 * @return void
	 *
	 * @throws Exception
	 *
	 * @since        1.0.0
	 *
	 * @noinspection PhpUnused
	 * @noinspection PhpUnusedParameterInspection
	 */
	public function onAjaxWishboxQuickOrderSubmitForm(AjaxEvent $event): void
	{
		require_once(JPATH_SITE . '/components/com_jshopping/bootstrap.php');
		require_once(JPATH_SITE . '/components/com_jshopping/loadparams.php');

		$app = $this->getApplication();

		$productId = $app->input->getInt('product_id', 0);
		$quantity = $app->input->getInt('quantity', 0);

		/** @var OrdersModel $ordersModel */
		$ordersModel= $this->getMVCFactory()
			->createModel('orders', 'Administrator');
		$ordersModel->setState('enable_check_step', $this->params->get('enable_check_step', 1));

		$paymentMethodId = (int) $this->params->get('payment_method_id',0);

		/** @var PaymentMethodTable $paymentMethodTable */
		$paymentMethodTable = JSFactory::getTable('paymentmethod');

		$paymentMethodTable->load($paymentMethodId);

		try
		{
			/** @var Dispatcher $dispatcher */
			$dispatcher = $this->traitGetDispatcher();

			$onApplicationBeforeRespond = function()
			{
				throw new Exception;
			};

			$dispatcher->addListener(
				ApplicationEvents::BEFORE_RESPOND,
				$onApplicationBeforeRespond
			);

			try
			{
				$ordersModel->create(
					$productId,
					$quantity,
					$paymentMethodTable->payment_class
				);
			}
			catch (Exception $e)
			{
				echo new JsonResponse(new Exception);
				die;
			}

			$dispatcher->removeListener(
				ApplicationEvents::BEFORE_RESPOND,
				$onApplicationBeforeRespond
			);

			$orderNumber = $ordersModel->state->get('order_number');
			$app->enqueueMessage(Text::sprintf('PLG_JSHOPPING_WISHBOXQUICKORDER_MESSAGE_ORDER_S_ACCEPTED', $orderNumber));
		}
		catch (Exception $e)
		{
			$app->enqueueMessage($e->getMessage());
		}
	}

	/**
	 * @param   Event  $event
	 *
	 * @return void
	 *
	 * @throws Exception
	 *
	 * @since        1.0.0
	 *
	 * @noinspection PhpUnused
	 */
	public function onBeforeCheckUserEmailExistJshopUserShop(Event $event): void
	{
		$query = $event->getArgument(4);

		$app = Factory::getApplication();
		$input = $app->getInput();

		if ($input->get('option') == 'com_ajax' && $input->get('plugin') == 'wishboxquickordersubmitform')
		{
			$query .= ' AND 1 <> 1';
		}

		$event->setArgument(4, $query);
	}

	/**
	 * @param   AfterRenderEvent $event  The event instance.
	 *
	 * @return  void
	 *
	 * @since 1.0.0
	 */
	public function onAfterRender(AfterRenderEvent $event): void
	{
		$app = $event->getApplication();

		if (!$app->isClient('site'))
		{
			return;
		}

		$style = $this->params->get('style', 'bootstrap');

		/** @var ConfigTable $jshopConfig */
		$jshopConfig = JSFactory::getConfig();

		$formHtml = LayoutHelper::render(
			'plugins.jshopping.wishboxquickorder.' . $style . '.form',
			[
				'jshopConfig'   => $jshopConfig,
				'params'        => $this->params,
			]
		);

		$buffer = $app->getBody();
		$buffer = str_replace('</body>', $formHtml . '</body>', $buffer);
		$app->setBody($buffer);
	}
}
