<?php
/*
 * @package   pkg_radicalmart_favorites
 * @version   1.0.0
 * @author    Dmitriy Vasyukov - https://fictionlabs.ru
 * @copyright Copyright (c) 2023 Fictionlabs. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://fictionlabs.ru/
 */

namespace Joomla\Component\RadicalMartFavorites\Administrator\Model;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\MVC\Model\ListModel;
use Joomla\Utilities\ArrayHelper;

class FavoritesModel extends ListModel
{
	/**
	 * Model context string.
	 *
	 * @var  string
	 *
	 * @since  1.0.0
	 */
	protected $context = 'com_radicalmart_favorites.favorites';

	/**
	 * The URL option for the component.
	 *
	 * @var    string
	 * @since  3.0
	 */
	protected $option = 'com_radicalmart_favorites';

	/**
	 * Constructor.
	 *
	 * @param   array  $config  An optional associative array of configuration settings.
	 *
	 * @since  1.0.0
	 */
	public function __construct($config = array())
	{
		// Add the ordering filtering fields whitelist
		if (empty($config['filter_fields']))
		{
			$config['filter_fields'] = array(
				'id', 'f.id',
				'product_id', 'f.product_id',
				'user_id', 'f.user_id',
				'date', 'f.date'
			);
		}

		parent::__construct($config);
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * @param   string  $ordering   An optional ordering field.
	 * @param   string  $direction  An optional direction (asc|desc).
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	protected function populateState($ordering = null, $direction = null)
	{
		// Set search filter state
		$search = $this->getUserStateFromRequest($this->context . '.filter.search', 'filter_search');
		$this->setState('filter.search', $search);

		// Set item_id filter state
		$item_id = $this->getUserStateFromRequest($this->context . '.filter.product_id', 'filter_product_id', '');
		$this->setState('filter.product_id', $item_id);

		// Set context filter state
		$context = $this->getUserStateFromRequest($this->context . '.filter.user_id', 'filter_user_id', '');
		$this->setState('filter.user_id', $context);

		// List state information
		$ordering  = empty($ordering) ? 'f.date' : $ordering;
		$direction = empty($direction) ? 'desc' : $direction;

		parent::populateState($ordering, $direction);
	}

	/**
	 * Method to get a store id based on model configuration state.
	 *
	 * @param   string  $id  A prefix for the store id.
	 *
	 * @return  string  A store id.
	 *
	 * @since  1.0.0
	 */
	protected function getStoreId($id = '')
	{
		$id .= ':' . $this->getState('filter.search');
		$id .= ':' . $this->getState('filter.product_id');
		$id .= ':' . $this->getState('filter.user_id');

		return parent::getStoreId($id);
	}

	/**
	 * Method to get a DatabaseQuery object for retrieving the data set from a database.
	 *
	 * @return  \Joomla\Database\QueryInterface  A QueryInterface object to retrieve the data set.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	protected function getListQuery()
	{
		$db    = $this->getDatabase();
		$query = $db->getQuery(true)
			->select(array('f.*'))
			->from($db->quoteName('#__radicalmart_favorites', 'f'));

		// Join over the users
		$query->select(array('u.name as user_name'))
			->leftJoin($db->quoteName('#__users', 'u') . ' ON u.id = f.user_id');

		// Join over the products
		$query->select(array('p.title as product_title'))
			->leftJoin($db->quoteName('#__radicalmart_products', 'p') . ' ON p.id = f.product_id');

		// Filter by product
		$product_id = $this->getState('filter.product_id');
		if (is_numeric($product_id))
		{
			$query->where('f.product_id = ' . (int) $product_id);
		}

		// Filter by user
		$user_id = $this->getState('filter.user_id');
		if ($user_id)
		{
			$query->where('r.user_id = ' . (int) $user_id);
		}

		$search = $this->getState('filter.search');
		if (!empty($search))
		{
			if (stripos($search, 'id:') === 0)
			{
				$query->where('f.id = ' . (int) substr($search, 3));
			}
			else if (stripos($search, 'product_id:') === 0)
			{
				$query->where('f.product_id = ' . (int) substr($search, 8));
			}
			else
			{
				$sql     = array();
				$columns = array('r.text', 'r.name', 'u.name', 'p.title');
				foreach ($columns as $column)
				{
					$sql[] = $db->quoteName($column) . ' LIKE '
						. $db->quote('%' . str_replace(' ', '%', $db->escape(trim($search), true) . '%'));
				}
				$query->where('(' . implode(' OR ', $sql) . ')');
			}
		}

		// Group by
		$query->group(array('f.id'));

		// Add the list ordering clause
		$ordering  = $this->state->get('list.ordering', 'f.date');
		$direction = $this->state->get('list.direction', 'desc');
		$query->order($db->escape($ordering) . ' ' . $db->escape($direction));

		return $query;
	}

	/**
	 * Method to delete one or more records.
	 *
	 * @param   array  &$pks  An array of record primary keys.
	 *
	 * @return  boolean  True if successful, false if an error occurs.
	 *
	 * @since   1.0.0
	 */
	public function delete(&$pks)
	{
		$pks   = ArrayHelper::toInteger((array) $pks);
		$table = $this->getTable();

		// Iterate the items to delete each one.
		foreach ($pks as $i => $pk)
		{
			if ($table->load($pk))
			{
				if (Factory::getApplication()->getIdentity()->authorise('core.delete', $this->option))
				{
					if (!$table->delete($pk))
					{
						$this->setError($table->getError());

						return false;
					}
				}
				else
				{
					// Prune items that you can't change.
					unset($pks[$i]);
					$error = $this->getError();

					if ($error)
					{
						Log::add($error, Log::WARNING, 'jerror');

						return false;
					}
					else
					{
						Log::add(Text::_('JLIB_APPLICATION_ERROR_DELETE_NOT_PERMITTED'), Log::WARNING, 'jerror');

						return false;
					}
				}
			}
			else
			{
				$this->setError($table->getError());

				return false;
			}
		}

		 // Clear the component's cache
        $this->cleanCache();

        return true;
	}
}