<?php
/*
 * @package   pkg_radicalmart_favorites
 * @version   1.0.0
 * @author    Dmitriy Vasyukov - https://fictionlabs.ru
 * @copyright Copyright (c) 2023 Fictionlabs. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://fictionlabs.ru/
 */

namespace Joomla\Component\RadicalMartFavorites\Site\Controller;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\LayoutHelper;
use Joomla\CMS\MVC\Controller\AdminController;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\Component\RadicalMart\Administrator\Traits\JsonControllerTrait;
use Joomla\Component\RadicalMartFavorites\Site\Helper\FavoritesHelper;

class FavoriteController extends AdminController
{
	use JsonControllerTrait;

	/**
	 * The URL option for the component.
	 *
	 * @var    string
	 *
	 * @since  1.0.0
	 */
	protected $option = 'com_radicalmart_favorites';

	/**
	 * The prefix to use with controller messages.
	 *
	 * @var  string
	 *
	 * @since  1.0.0
	 */
	protected $text_prefix = 'COM_RADICALMART_FAVORITES';

	/**
	 * Proxy for getModel.
	 *
	 * @param   string  $name    The model name.
	 * @param   string  $prefix  The class prefix.
	 * @param   array   $config  The array of possible config values.
	 *
	 * @return  BaseDatabaseModel  A model object.
	 *
	 * @since  1.0.0
	 */
	public function getModel($name = 'Favorite', $prefix = 'Administrator', $config = array('ignore_request' => true))
	{
		return parent::getModel($name, $prefix, $config);
	}

	/**
	 * Method to add product to favorite.
	 *
	 * @return  bool True on success, False on failure.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function toggle()
	{
		$app        = Factory::getApplication();
		$product_id = $this->app->input->getInt('product_id');
		$state      = $this->app->input->getInt('state', 0);
		$simple     = $this->app->input->getInt('simple', 0);
		$context    = $this->app->input->getString('context');
		$userId     = $app->getIdentity()->id;

		// Save to database for registered users
		if ($userId)
		{
			$method = 'save';

			if (!$product_id)
			{
				$this->code    = 500;
				$this->message = Text::_('COM_RADICALMART_FAVORITES_ERROR_EMPTY_PRODUCT');

				return $this->setJsonResponse();
			}

			$model = $this->getModel();
			$data  = [
				'product_id' => $product_id
			];

			if ($state === 1)
			{
				$method = 'delete';
				$data   = FavoritesHelper::getFavoriteByProductId($product_id);
			}

			// Save review
			if (!$favoriteId = $model->{$method}($data))
			{
				$this->message = [];
				foreach ($model->getErrors() as $error)
				{
					$this->message[] = ($error instanceof \Exception) ? $error->getMessage() : $error;
				}
				$this->code = 500;

				return $this->setJsonResponse();
			}
		}

		$favorites = $app->getUserState('favorites_items_' . $userId, []);

		if (($key = array_search($product_id, $favorites)) !== false)
		{
			unset($favorites[$key]);
		}
		else
		{
			$favorites[] = $product_id;
		}

		$app->setUserState('favorites_items_empty_' . $userId, false);
		$app->setUserState('favorites_items_' . $userId, $favorites);

		$layout = 'add';

		if (!$state)
		{
			if ($simple)
			{
				$layout = 'remove';
			}
			else
			{
				$layout = 'dropdown';
			}
		}

		return $this->setJsonResponse(
			[
				'message'  => Text::_('COM_RADICALMART_FAVORITE_SUCCESS'),
				'html'     => LayoutHelper::render('components.radicalmart_favorites.buttons.toggle.' . $layout, ['product_id' => $product_id, 'active' => !$state, 'context' => $context]),
				'redirect' => false
			]
		);
	}

	/**
	 * Method to add product to favorite.
	 *
	 * @return  bool True on success, False on failure.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function count()
	{
		$count = FavoritesHelper::getFavoritesCount();

		return $this->setJsonResponse($count);
	}
}