<?php
/*
 * @package   pkg_radicalmart_favorites
 * @version   1.0.0
 * @author    Dmitriy Vasyukov - https://fictionlabs.ru
 * @copyright Copyright (c) 2023 Fictionlabs. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://fictionlabs.ru/
 */

namespace Joomla\Plugin\RadicalMart\Favorites\Extension;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Layout\LayoutHelper;
use Joomla\CMS\Menu\AdministratorMenuItem;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Component\RadicalMartFavorites\Site\Helper\FavoritesHelper;
use Joomla\Event\DispatcherInterface;
use Joomla\Event\Event;
use Joomla\Event\SubscriberInterface;
use Joomla\Registry\Registry;

class Favorites extends CMSPlugin implements SubscriberInterface
{
	/**
	 * Load the language file on instantiation.
	 *
	 * @var    bool
	 *
	 * @since  1.0.0
	 */
	protected $autoloadLanguage = true;

	/**
	 * Loads the application object.
	 *
	 * @var  \Joomla\CMS\Application\CMSApplication
	 *
	 * @since  1.0.0
	 */
	protected $app = null;

	/**
	 * Loads the database object.
	 *
	 * @var  \Joomla\Database\DatabaseDriver
	 *
	 * @since  1.0.0
	 */
	protected $db = null;

	/**
	 * RadicalMart component params.
	 *
	 * @var  Registry|null
	 *
	 * @since  1.0.0
	 */
	protected ?Registry $componentParams = null;

	/**
	 * Constructor.
	 *
	 * @param   DispatcherInterface  &$subject  The object to observe.
	 * @param   array                 $config   An optional associative array of configuration settings.
	 *
	 * @since  1.0.0
	 */
	public function __construct(&$subject, $config = [])
	{
		parent::__construct($subject, $config);

		$this->componentParams = ComponentHelper::getParams('com_radicalmart');
	}

	/**
	 * Returns an array of events this subscriber will listen to.
	 *
	 * @return  array
	 *
	 * @since   1.0.0
	 */
	public static function getSubscribedEvents(): array
	{
		return [
			'onBeforeRender'                 => 'onBeforeRender',
			'onRadicalMartPreprocessSubmenu' => 'onRadicalMartPreprocessSubmenu',
			'onRadicalMartPrepareConfigForm' => 'onRadicalMartPrepareConfigForm',
			'onContentAfterTitle'            => 'onContentAfterTitle',
			'onContentBeforeDisplay'         => 'onContentBeforeDisplay',
			'onContentAfterDisplay'          => 'onContentAfterDisplay'
		];
	}

	/**
	 * Method to change forms.
	 *
	 * @param   Form   $form  The form to be altered.
	 * @param   mixed  $data  The associated data for the form.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function onRadicalMartPrepareConfigForm(Form $form, $data)
	{
		Form::addFormPath(JPATH_PLUGINS . '/' . $this->_type . '/' . $this->_name . '/forms');
		$form->loadFile('config');
	}

	/**
	 * Listener for the `onBeforeRender` event.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function onBeforeRender()
	{
		if ($this->app->isClient('administrator')
			&& $this->app->input->getCmd('option') === 'com_config'
			&& $this->app->input->getCmd('view') === 'component'
			&& $this->app->input->getCmd('component') === 'com_radicalreviews')
		{
			$assets = $this->app->getDocument()->getWebAssetManager();
			$assets->getRegistry()->addExtensionRegistryFile('com_radicalreviews');
			$assets->useScript('com_radicalreviews.administrator.config');
		}
	}

	/**
	 * The display event.
	 *
	 * @param   Event  $event  The event.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function onContentAfterTitle(Event $event)
	{
		if ($this->componentParams->get('registered_only', 1) && Factory::getApplication()->getIdentity()->guest)
		{
			return;
		}

		$context = $event->getArgument(0);
		$item    = $event->getArgument(1);
		$params  = $event->getArgument(2);
		$result  = $event->getArgument('result', []);

		$result[] = $this->display($context, $item, $params, 1);

		$event->setArgument('result', $result);
	}

	/**
	 * The display event.
	 *
	 * @param   Event  $event  The event.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function onContentBeforeDisplay(Event $event)
	{
		$context = $event->getArgument(0);
		$item    = $event->getArgument(1);
		$params  = $event->getArgument(2);
		$result  = $event->getArgument('result', []);

		$result[] = $this->display($context, $item, $params, 2);

		$event->setArgument('result', $result);
	}

	/**
	 * The display event.
	 *
	 * @param   Event  $event  The event.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function onContentAfterDisplay(Event $event)
	{
		$context = $event->getArgument(0);
		$item    = $event->getArgument(1);
		$params  = $event->getArgument(2);
		$result  = $event->getArgument('result', []);

		$result[] = $this->display($context, $item, $params, 3);

		$event->setArgument('result', $result);
	}

	/**
	 * Performs the display event.
	 *
	 * @param   string     $context      The context
	 * @param   \stdClass  $item         The item
	 * @param   Registry   $params       The params
	 * @param   integer    $displayType  The type
	 *
	 * @return  string
	 *
	 * @since   1.0.0
	 */
	private function display($context, $item, $params, $displayType)
	{
		if ($this->app->isClient('site') && strpos($context, 'radicalmart') !== false)
		{
			$result = array();

			// Display stats
			if ($displayType === (int) $this->componentParams->get('favorites_display', 0))
			{
				$active   = FavoritesHelper::checkActive($item->id);
				$result[] = LayoutHelper::render('components.radicalmart_favorites.buttons.toggle', ['product_id' => $item->id, 'active' => $active, 'context' => $context]);
			}

			return implode("\n", $result);
		}
	}

	/**
	 * Method to add submenu items to RadicalMart sidebar menu
	 *
	 * @param   array                  $results
	 * @param   AdministratorMenuItem  $parent
	 * @param   Registry               $params
	 *
	 * @throws \Exception
	 *
	 * @since 1.0.0
	 */
	public function onRadicalMartPreprocessSubmenu(array &$results, AdministratorMenuItem $parent, Registry $params)
	{
		Factory::getApplication()->getLanguage()->load('com_radicalmart_favorites', JPATH_ADMINISTRATOR);

		$results[] = new AdministratorMenuItem (array(
			'title'     => 'COM_RADICALMART_FAVORITE',
			'type'      => 'component',
			'link'      => 'index.php?option=com_radicalmart_favorites&view=favorites',
			'element'   => 'com_radicalmart_favorites',
			'class'     => 'icon-heart',
			'ajaxbadge' => '',
			'dashboard' => '',
			'scope'     => 'default',
		));
	}
}