<?php
/*
 * @package   pkg_radicalmart_favorites
 * @version   1.0.1
 * @author    Dmitriy Vasyukov - https://fictionlabs.ru
 * @copyright Copyright (c) 2023 Fictionlabs. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://fictionlabs.ru/
 */

namespace Joomla\Component\RadicalMartFavorites\Site\Model;

defined('_JEXEC') or die;

use Joomla\CMS\Application\SiteApplication;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Multilanguage;
use Joomla\CMS\MVC\Model\ListModel;
use Joomla\CMS\Pagination\Pagination;

class FavoritesModel extends ListModel
{
	/**
	 * Model context string.
	 *
	 * @var  string
	 *
	 * @since  1.0.0
	 */
	protected $context = 'com_radicalmart_favorites.favorites';

	/**
	 * The URL option for the component.
	 *
	 * @var    string
	 * @since  3.0
	 */
	protected $option = 'com_radicalmart_favorites';

	/**
	 * Pagination object.
	 *
	 * @var  Pagination
	 *
	 * @since  1.0.0
	 */
	protected $_pagination;

	/**
	 * Method to auto-populate the model state.
	 *
	 * @param   string  $ordering   An optional ordering field.
	 * @param   string  $direction  An optional direction (asc|desc).
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	protected function populateState($ordering = null, $direction = null)
	{
		/* @var SiteApplication $app */
		$app = Factory::getApplication('site');

		// Set params state
		$params = ComponentHelper::getParams('com_radicalmart');
		$params->merge($app->getParams());
		$this->setState('params', $params);

		// Set published && debug state
		if ($app->input->getInt('debug', 0))
		{
			$this->setState('filter.published', array(0, 1));
			$this->setState('debug', 1);
		}
		else
		{
			$this->setState('filter.published', 1);
		}

		// Set language filter state
		$this->setState('filter.language', Multilanguage::isEnabled());

		// List state information
		parent::populateState($ordering, $direction);

		// Set ordering for query
		$this->setState('list.ordering', $ordering);
		$this->setState('list.direction', $direction);

		// Set limit & start for query
		$this->setState('list.limit', (int) $params->get('favorites_limit', 9));
		$this->setState('list.start', $app->input->get('start', 0, 'uint'));
	}

	/**
	 * Method to get a store id based on model configuration state.
	 *
	 * @param   string  $id  A prefix for the store id.
	 *
	 * @return  string  A store id.
	 *
	 * @since  1.0.0
	 */
	protected function getStoreId($id = '')
	{
		$id .= ':' . serialize($this->getState('filter.published'));
		$id .= ':' . $this->getState('filter.language');

		return parent::getStoreId($id);
	}

	/**
	 * Method to get a favorite items.
	 *
	 * @return  array  Array of products ids.
	 *
	 * @since  1.0.0
	 */
	protected function getFavorites()
	{
		$userId = Factory::getApplication()->getIdentity()->id;

		if (!$userId)
		{
			return Factory::getApplication()->getUserState('favorites_items_' . $userId, []);
		}

		$db     = $this->getDatabase();
		$query  = $db->getQuery(true);
		$query->select('product_id')
			->from($db->quoteName('#__radicalmart_favorites'))
			->where($db->quoteName('user_id') . ' = ' . $userId);

		$db->setQuery($query);

		return $db->loadColumn();
	}


	/**
	 * Get the favorites.
	 *
	 * @return  mixed  An array of objects or false if an error occurs.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function getItems()
	{
		$favorites = $this->getFavorites();


		if (!$favorites)
		{
			return array();
		}

		$model = Factory::getApplication()->bootComponent('com_radicalmart')->getMVCFactory()->createModel('Products', 'Site', ['ignore_request' => true]);
		$model->setState('params', $this->getState('params'));
		$model->setState('filter.item_id', $favorites);
		$model->setState('filter.published', $this->getState('filter.published'));
		$model->setState('filter.language', $this->getState('filter.language'));
		$model->setState('list.limit', $this->getState('list.limit'));
		$model->setState('list.start', $this->getState('list.start'));

		$products = $model->getItems();

		if ($products === false)
		{
			$this->setError($model->getError());
		}

		$this->_pagination = $model->getPagination();

		return $products;
	}

	/**
	 * Method to get a JPagination object for the data set.
	 *
	 * @return  Pagination  A Pagination object for the data set.
	 *
	 * @since  1.0.0
	 */
	public function getPagination()
	{
		return (empty($this->_pagination)) ? null : $this->_pagination;
	}
}