<?php
/*
 * @package   pkg_radicalmart_favorites
 * @version   1.0.1
 * @author    Dmitriy Vasyukov - https://fictionlabs.ru
 * @copyright Copyright (c) 2023 Fictionlabs. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://fictionlabs.ru/
 */

namespace Joomla\Component\RadicalMartFavorites\Site\View\Favorites;

defined('_JEXEC') or die;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Helper\ModuleHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\View\HtmlView as BaseHtmlView;
use Joomla\CMS\Object\CMSObject;
use Joomla\CMS\Pagination\Pagination;
use Joomla\Registry\Registry;

class HtmlView extends BaseHtmlView
{
	/**
	 * Model state variables.
	 *
	 * @var  CMSObject
	 *
	 * @since  1.0.0
	 */
	protected $state;

	/**
	 * Application params.
	 *
	 * @var  Registry;
	 *
	 * @since  1.0.0
	 */
	public $params;

	/**
	 * This view canonical link.
	 *
	 * @var  string
	 *
	 * @since  1.0.0
	 */
	protected $link;

	/**
	 * An array of items.
	 *
	 * @var  array
	 *
	 * @since  1.0.0
	 */
	protected $items;

	/**
	 * Pagination object.
	 *
	 * @var  Pagination
	 *
	 * @since  1.0.0
	 */
	protected $pagination;

	/**
	 * RadicalMart mode.
	 *
	 * @var  string
	 *
	 * @since  1.0.0
	 */
	protected $mode;

	/**
	 * Page class suffix from params.
	 *
	 * @var  string
	 *
	 * @since  1.0.0
	 */
	public $pageclass_sfx;

	/**
	 * Modules list for current view.
	 *
	 * @var  array
	 *
	 * @since  1.0.0
	 */
	protected $modules;

	/**
	 * Display the view.
	 *
	 * @param   string  $tpl  The name of the template file to parse.
	 *
	 * @return  mixed  A string if successful, otherwise an Error object.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function display($tpl = null)
	{
		$this->state      = $this->get('State');
		$this->items      = $this->get('Items');
		$this->pagination = $this->get('Pagination');
		$this->params     = $this->state->params;
		$this->app        = Factory::getApplication();

		// Check for errors
		if (count($errors = $this->get('Errors'))) throw new \Exception(implode('\n', $errors), 500);

		// Set mode
		$this->mode = ComponentHelper::getParams('com_radicalmart')->get('mode', 'shop');

		// Escape strings for html output
		$this->pageclass_sfx = htmlspecialchars($this->params->get('pageclass_sfx', ''));

		// Get modules
		$this->modules = array();
		$positions     = array();
		$this->app->triggerEvent('onRadicalMartGetProductModulesPositions', array('com_radicalmart_favorites.favorites', &$positions));
		foreach ($positions as $position)
		{
			foreach (ModuleHelper::getModules($position) as $module)
			{
				$module->sorceShowtitle = $module->showtitle;
				$module->showtitle      = 0;
				$module->params         = new Registry($module->params);
				$module->params->set('style', 'none');
				$module->params = (string) $module->params;
				$render         = ModuleHelper::renderModule($module);
				if (!empty(trim($render)))
				{
					$module->render = $render;
					if (!isset($this->modules[$position])) $this->modules[$position] = array();

					$this->modules[$position][] = $module;
				}
				$module->showtitle = $module->sorceShowtitle;
			}
		}

		// Load language
		$this->app->getLanguage()->load('com_radicalmart');

		// Prepare the document
		$this->_prepareDocument();

		parent::display($tpl);
	}

	/**
	 * Prepare the document.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	protected function _prepareDocument()
	{
		// Because the application sets a default page title,
		// we need to get it from the menu item itself
		$menu = Factory::getApplication()->getMenu()->getActive();

		if ($menu)
		{
			$this->params->def('page_heading', $this->params->get('page_title', $menu->title));
		}
		else
		{
			$this->params->def('page_heading', Text::_('COM_RADICALMART_FAVORITES_VIEW_REQUEST_PAGE_TITLE'));
		}

		$this->setDocumentTitle($this->params->get('page_title', ''));

		if ($this->params->get('menu-meta_description'))
		{
			$this->document->setDescription($this->params->get('menu-meta_description'));
		}

		if ($this->params->get('robots'))
		{
			$this->document->setMetaData('robots', $this->params->get('robots'));
		}
	}
}