<?php
/*
 * @package   pkg_radicalmart_favorites
 * @version   2.0.0
 * @author    Dmitriy Vasyukov - https://fictionlabs.ru
 * @copyright Copyright (c) 2026 Fictionlabs. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://fictionlabs.ru/
 */

namespace Joomla\Component\RadicalMartFavorites\Site\Helper;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Helper\RouteHelper as RouteHelperCore;
use Joomla\CMS\Layout\LayoutHelper;
use Joomla\Component\RadicalMart\Administrator\Helper\ParamsHelper;

class FavoritesHelper extends RouteHelperCore
{
	/**
	 * Method for check active button state.
	 *
	 * @param   string  $id  The product id.
	 *
	 * @return  bool
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public static function checkActive(int $id = 0)
	{
		$app    = Factory::getApplication();
		$userId = $app->getIdentity()->id;
		$empty  = $app->getUserState('favorites_items_empty_' . $userId, false);

		// Unactive
		if ($empty) return false;

		$favorites = $app->getUserState('favorites_items_' . $userId);

		if (empty($favorites))
		{
			$userId = $app->getIdentity()->id;
			$db     = Factory::getContainer()->get('DatabaseDriver');
			$query  = $db->getQuery(true);
			$query->select('item_id')
				->from($db->quoteName('#__radicalmart_favorites'))
				->where($db->quoteName('user_id') . ' = ' . $userId);

			$db->setQuery($query);

			$favorites = $db->loadColumn();

			// Set empty state
			if (empty($favorites))
			{
				$app->setUserState('favorites_items_empty_' . $userId, true);
			}

			// Set favorite list to state
			$app->setUserState('favorites_items_' . $userId, $favorites);
		}

		if (in_array($id, $favorites))
		{
			return true;
		}

		return false;
	}

	/**
	 * Method for render favorites button.
	 *
	 * @param   string  $pk       The item id.
	 * @param   string  $context  Contenxt
	 * @param   bool    $active   Active item flag
	 *
	 * @return  string
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public static function renderButton(int $pk = 0, string $context = 'com_radicalmart.product', int $active = null): string
	{
		$params = ParamsHelper::getComponentParams();

		if ($params->get('registered_only', 1) && Factory::getApplication()->getIdentity()->guest)
			return '';

		if (is_null($active)) $active = self::checkActive($pk);

		// Load language
		Factory::getApplication()->getLanguage()->load('com_radicalmart_favorites');

		return LayoutHelper::render('components.radicalmart_favorites.buttons.toggle', ['product_id' => $pk, 'active' => $active, 'context' => $context]);
	}

	/**
	 * Method for get favorite item by favorite id.
	 *
	 * @param   int  $item_id  The item id.
	 *
	 * @return  bool
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public static function getFavoriteByItemId(int $item_id = 0)
	{
		$app    = Factory::getApplication();
		$userId = $app->getIdentity()->id;
		$db     = Factory::getContainer()->get('DatabaseDriver');
		$query  = $db->getQuery(true);
		$query->select('id')
			->from($db->quoteName('#__radicalmart_favorites'))
			->where($db->quoteName('user_id') . ' = ' . $userId)
			->where($db->quoteName('item_id') . ' = ' . $item_id);

		$db->setQuery($query);

		$favorite = $db->loadResult();

		return $favorite;
	}

	/**
	 * Method for get favorites count.
	 *
	 * @return  int
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public static function getFavoritesCount()
	{
		$app    = Factory::getApplication();
		$userId = $app->getIdentity()->id;
		$empty  = $app->getUserState('favorites_items_empty_' . $userId, false);

		// Check empty state
		if ($empty)
		{
			return 0;
		}

		// Get favorites from user state
		$favorites = $app->getUserState('favorites_items_' . $userId);

		if ($favorites)
		{
			return count($favorites);
		}

		$db    = Factory::getContainer()->get('DatabaseDriver');
		$query = $db->getQuery(true);
		$query->select('COUNT(*)')
			->from($db->quoteName('#__radicalmart_favorites'))
			->where($db->quoteName('user_id') . ' = ' . $userId);

		$db->setQuery($query);

		$count = $db->loadResult();

		return $count;
	}
}