<?php
/*
 * @package   pkg_radicalmart_favorites
 * @version   2.0.0
 * @author    Dmitriy Vasyukov - https://fictionlabs.ru
 * @copyright Copyright (c) 2026 Fictionlabs. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://fictionlabs.ru/
 */

namespace Joomla\Plugin\RadicalMart\Favorites\Extension;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Menu\AdministratorMenuItem;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Component\RadicalMartFavorites\Site\Helper\FavoritesHelper;
use Joomla\Database\QueryInterface;
use Joomla\Event\DispatcherInterface;
use Joomla\Event\SubscriberInterface;
use Joomla\Registry\Registry;

class Favorites extends CMSPlugin implements SubscriberInterface
{
	/**
	 * Points enabled.
	 *
	 * @var  bool
	 *
	 * @since  1.0.0
	 */
	protected bool $favorites = false;

	/**
	 * Load the language file on instantiation.
	 *
	 * @var    bool
	 *
	 * @since  1.0.0
	 */
	protected $autoloadLanguage = true;

	/**
	 * Loads the application object.
	 *
	 * @var  \Joomla\CMS\Application\CMSApplication
	 *
	 * @since  1.0.0
	 */
	protected $app = null;

	/**
	 * Loads the database object.
	 *
	 * @var  \Joomla\Database\DatabaseDriver
	 *
	 * @since  1.0.0
	 */
	protected $db = null;

	/**
	 * RadicalMart component params.
	 *
	 * @var  Registry|null
	 *
	 * @since  1.0.0
	 */
	protected ?Registry $componentParams = null;

	/**
	 * Constructor.
	 *
	 * @param   DispatcherInterface  &$subject  The object to observe.
	 * @param   array                 $config   An optional associative array of configuration settings.
	 *
	 * @since  1.0.0
	 */
	public function __construct(&$subject, $config = [])
	{
		parent::__construct($subject, $config);

		$this->componentParams = ComponentHelper::getParams('com_radicalmart');
	}

	/**
	 * Returns an array of events this subscriber will listen to.
	 *
	 * @return  array
	 *
	 * @since   1.0.0
	 */
	public static function getSubscribedEvents(): array
	{
		return [
			'onBeforeRender'                       => 'onBeforeRender',
			'onRadicalMartPreprocessSubmenu'       => 'onRadicalMartPreprocessSubmenu',
			'onRadicalMartPrepareConfigForm'       => 'onRadicalMartPrepareConfigForm',
			'onRadicalMartPrepareConfigGroups'     => 'onRadicalMartPrepareConfigGroups',
			'onRadicalMartPreprocessSiteListQuery' => 'onRadicalMartPreprocessSiteListQuery',
			'onRadicalMartGetMenuTypeOptions'      => 'onRadicalMartGetMenuTypeOptions',

			'onContentAfterTitle'    => 'onContentAfterTitle',
			'onContentBeforeDisplay' => 'onContentBeforeDisplay',
			'onContentAfterDisplay'  => 'onContentAfterDisplay',
		];
	}

	/**
	 * Listener for `onRadicalMartGetMenuTypeOptions` event.
	 *
	 * @param   array  $parent  RadicalMart menu items.
	 * @param   array  $list    All menu items list.
	 *
	 * @since 1.0.0
	 */
	public function onRadicalMartGetMenuTypeOptions(array &$parent = [], array &$list = [])
	{
		if (empty($list['COM_RADICALMART_FAVORITES']))
		{
			return;
		}

		foreach ($list['COM_RADICALMART_FAVORITES'] as $item)
		{
			if (!isset($item->request['layout']))
			{
				$parent[] = $item;
			}
		}

		unset($list['COM_RADICALMART_FAVORITES']);
	}

	/**
	 * Trigger `onRadicalMartPreprocessSiteListQuery` event.
	 *
	 * @param   string          $context  Current context.
	 * @param   Registry        $state    Current model states.
	 * @param   QueryInterface  $query    Modification query object
	 *
	 * @return void
	 */
	public function onRadicalMartPreprocessSiteListQuery(string $context, Registry $state, QueryInterface &$query): void
	{
		if ($this->app->isClient('administrator') || !$state->get('filter.favorites')) return;

		$favorites = $state->get('filter.item_id');

		if ($favorites)
		{
			$db = Factory::getContainer()->get('DatabaseDriver');

			// Clear order
			$query->clear('order');
			$query->order('FIELD(' . $db->quoteName('p.id') . ', ' . implode(',', $favorites) . ')');
		}
	}

	/**
	 * Method to change forms.
	 *
	 * @param   Form   $form  The form to be altered.
	 * @param   mixed  $data  The associated data for the form.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function onRadicalMartPrepareConfigForm(Form $form, $data)
	{
		Form::addFormPath(JPATH_PLUGINS . '/' . $this->_type . '/' . $this->_name . '/forms');
		$form->loadFile('config');
	}

	/**
	 * Listener for the `onBeforeRender` event.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function onBeforeRender()
	{
		if ($this->app->isClient('administrator')
			&& $this->app->input->getCmd('option') === 'com_config'
			&& $this->app->input->getCmd('view') === 'component'
			&& $this->app->input->getCmd('component') === 'com_radicalreviews')
		{
			$assets = $this->app->getDocument()->getWebAssetManager();
			$assets->getRegistry()->addExtensionRegistryFile('com_radicalmart_favorites');
			$assets->useScript('com_radicalmart_favorites.administrator.config');
		}
	}

	/**
	 * The display event.
	 *
	 * @param   mixed  $event  The event.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function onContentAfterTitle($event)
	{
		$context = $event->getArgument(0);
		$item    = $event->getArgument(1);
		$params  = $event->getArgument(2);
		$result  = $event->getArgument('result', []);

		$result['radicalmart_favorites'] = $this->display($context, $item, $params, 1);

		$event->setArgument('result', $result);
	}

	/**
	 * The display event.
	 *
	 * @param   mixed  $event  The event.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function onContentBeforeDisplay($event)
	{
		$context = $event->getArgument(0);
		$item    = $event->getArgument(1);
		$params  = $event->getArgument(2);
		$result  = $event->getArgument('result', []);

		$result['radicalmart_favorites'] = $this->display($context, $item, $params, 2);

		$event->setArgument('result', $result);
	}

	/**
	 * The display event.
	 *
	 * @param   mixed  $event  The event.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function onContentAfterDisplay($event)
	{
		$context = $event->getArgument(0);
		$item    = $event->getArgument(1);
		$params  = $event->getArgument(2);
		$result  = $event->getArgument('result', []);

		$result['radicalmart_favorites'] = $this->display($context, $item, $params, 3);

		$event->setArgument('result', $result);
	}

	/**
	 * Performs the display event.
	 *
	 * @param   string     $context      The context
	 * @param   \stdClass  $item         The item
	 * @param   Registry   $params       The params
	 * @param   integer    $displayType  The type
	 *
	 * @return  string
	 *
	 * @since   1.0.0
	 */
	private function display($context, $item, $params, $displayType)
	{
		if ($this->app->isClient('site') && strpos($context, 'radicalmart') !== false)
		{
			// Display stats
			if ($displayType === (int) $this->componentParams->get('favorites_display', 0))
			{
				return FavoritesHelper::renderButton($item->id, $context);
			}
		}

		return '';
	}

	/**
	 * Method to add fields to form.
	 *
	 * @param   array  $groups  The associated data for the form.
	 *
	 * @return void
	 *
	 * @throws \Exception
	 *
	 * @since 1.1.0
	 */
	public function onRadicalMartPrepareConfigGroups(array &$groups): void
	{
		Factory::getApplication()->getLanguage()->load('com_radicalmart_favorites', JPATH_ADMINISTRATOR);

		$groups['favorites'] = [
			'title'    => 'COM_RADICALMART_FAVORITES_CONFIG_TITLE',
			'key'      => 'favorites',
			'sections' => [
				'settings'    => [
					'title'     => 'COM_RADICALMART_FAVORITES_CONFIG_SETTINGS',
					'key'       => 'favorites_settings',
					'type'      => 'fieldsets',
					'fieldsets' => [
						'favorites_global', 'favorites_page'
					],
				],
				'permissions' => [
					'title'     => 'COM_RADICALMART_FAVORITES_CONFIG_PERMISSIONS',
					'key'       => 'favorites-permissions',
					'available' => ['access' => ['com_radicalmart_favorites' => 'core.admin']],
					'type'      => 'fieldsets',
					'fieldsets' => [
						'favorites_permissions'
					],
				],
			]
		];
	}

	/**
	 * Method to add submenu items to RadicalMart sidebar menu
	 *
	 * @param   array                  $results
	 * @param   AdministratorMenuItem  $parent
	 * @param   Registry               $params
	 *
	 * @throws \Exception
	 *
	 * @since 1.0.0
	 */
	public function onRadicalMartPreprocessSubmenu(array &$results, AdministratorMenuItem $parent, Registry $params)
	{
		Factory::getApplication()->getLanguage()->load('com_radicalmart_favorites', JPATH_ADMINISTRATOR);

		$results[] = new AdministratorMenuItem (array(
			'title'     => 'COM_RADICALMART_MENU_FAVORITES',
			'type'      => 'component',
			'link'      => 'index.php?option=com_radicalmart_favorites&view=favorites',
			'element'   => 'com_radicalmart_favorites',
			'class'     => 'icon-heart',
			'ajaxbadge' => '',
			'dashboard' => '',
			'scope'     => 'default',
		));
	}
}