<?php
/*
 * @package   pkg_radicalreviews
 * @version   1.1.0
 * @author    Delo Design
 * @copyright Copyright (c) 2023 Delo Design. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://delo-design.ru
 */

namespace Joomla\Component\RadicalReviews\Site\View\Review;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Filesystem\File;
use Joomla\CMS\Filesystem\Path;
use Joomla\CMS\HTML\Helpers\StringHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Menu\MenuItem;
use Joomla\CMS\MVC\View\HtmlView as BaseHtmlView;
use Joomla\CMS\Pagination\Pagination;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;
use Joomla\Component\RadicalReviews\Site\Helper\ReviewsHelper;
use Joomla\Component\RadicalReviews\Site\Helper\RouteHelper;
use Joomla\Registry\Registry;

class HtmlView extends BaseHtmlView
{
	/**
	 * Model state variables.
	 *
	 * @var  \Joomla\CMS\Object\CMSObject
	 *
	 * @since   1.0.0
	 */
	protected $state;

	/**
	 * Application params.
	 *
	 * @var  Registry
	 *
	 * @since   1.0.0
	 */
	public $params;

	/**
	 * Object.
	 *
	 * @var  object|false
	 *
	 * @since  1.0.0
	 */
	protected $object;

	/**
	 * Comment object.
	 *
	 * @var  object
	 *
	 * @since   1.0.0
	 */
	protected $item;

	/**
	 * Active menu item.
	 *
	 * @var  MenuItem
	 *
	 * @since  1.0.0
	 */
	protected $menu;

	/**
	 * Is current menu item.
	 *
	 * @var  bool
	 *
	 * @since  1.0.0
	 */
	protected $menuCurrent;

	/**
	 * Page class suffix from params.
	 *
	 * @var  string
	 *
	 * @since  1.0.0
	 */
	public $pageclass_sfx;

	/**
	 * Display the view.
	 *
	 * @param   string  $tpl  The name of the template file to parse.
	 *
	 * @return  mixed  A string if successful, otherwise an Error object.
	 *
	 * @throws  \Exception
	 *
	 * @since   1.0.0
	 */
	public function display($tpl = null)
	{
		$app          = Factory::getApplication();
		$this->state  = $this->get('State');
		$this->params = $this->state->get('params');
		$this->item   = $this->get('Item');
		$this->object = $this->get('Object');

		// Check object state
		if (!$this->object || $this->object->state !== 1)
		{
			throw new \Exception(Text::_('COM_RADICALREVIEWS_ERROR_OBJECT_NOT_FOUND'), 404);
		}

		// Check for errors
		if (count($errors = $this->get('Errors')))
		{
			throw new \Exception(implode('\n', $errors), 500);
		}

		// Check multilanguage redirect
		ReviewsHelper::checkMultilanguage($this->object);

		// Check review doubles
		$this->checkReviewDoubles();

		// Check reviewed state
		$key            = 'new_object_review_' . $this->state->get('object.context') . '_' . $this->state->get('object.item_id');
		$this->reviewed = $app->getUserState($key);

		// Load plugin language

		$app->getLanguage()->load('plg_radicalreviews_' . explode('.' . $this->object->context, 1)[0], JPATH_ADMINISTRATOR);

		// Prepare the document
		$this->_prepareDocument();

		parent::display($tpl);
	}

	/**
	 * Prepare the document.
	 *
	 * @throws  \Exception
	 *
	 * @since   1.0.0
	 */
	protected function _prepareDocument()
	{
		$app    = Factory::getApplication();
		$object = $this->object;

		$paths   = array(array('title' => Text::_('COM_RADICALREVIEWS_REVIEW'), 'link' => ''));
		$paths[] = array('title' => $this->object->title, 'link' => $this->object->link);

		// Change breadcrumbs
		$pathway = $app->getPathway();

		$pathwayObject       = new \stdClass();
		$pathwayObject->name = $this->object->title;
		$pathwayObject->link = $this->object->link;
		$pathwayItems[]      = $pathwayObject;

		$pathwayAllReviews       = new \stdClass();
		$pathwayAllReviews->name = Text::_('COM_RADICALREVIEWS_REVIEWS');
		$pathwayAllReviews->link = RouteHelper::getReviewsRoute($this->object->id, $this->object->context);
		$pathwayItems[]          = $pathwayAllReviews;

		$pathwayItem       = new \stdClass();
		$pathwayItem->name = Text::sprintf('COM_RADICALREVIEWS_REVIEW_SHORT_TITLE', $this->state->get('review.id'));
		$pathwayItem->link = '';
		$pathwayItems[]    = $pathwayItem;

		$pathway->setPathway($pathwayItems);

		// Set meta title
		$review   = $this->item;
		$title    = Text::sprintf('COM_RADICALREVIEWS_REVIEW_FULL_TITLE', $object->title, $review->id);
		$sitename = $app->get('sitename');
		if ($app->get('sitename_pagetitles', 0) == 1) $title = Text::sprintf('JPAGETITLE', $sitename, $title);
		elseif ($app->get('sitename_pagetitles', 0) == 2) $title = Text::sprintf('JPAGETITLE', $title, $sitename);
		$this->document->setTitle($title);

		// Set description
		$this->document->setDescription(StringHelper::truncate($review->text, 150, false, false));

		// Set meta image
		$image = $review->image ?? $object->image;
		$this->document->setMetadata('image', Uri::root() . $image);

		// Set meta robots
		if ($this->state->get('debug', 0))
		{
			$this->document->setMetadata('robots', 'noindex');
		}
		elseif ($this->params->get('robots'))
		{
			$this->document->setMetadata('robots', $this->params->get('robots'));
		}

		// Set meta url
		$url = Uri::getInstance()->toString(array('scheme', 'host', 'port')) . $review->link;
		$this->document->setMetaData('url', $url);

		// Set meta twitter
		$this->document->setMetaData('twitter:card', 'summary_large_image');
		$this->document->setMetaData('twitter:site', $sitename);
		$this->document->setMetaData('twitter:creator', $sitename);
		$this->document->setMetaData('twitter:title', $title);
		$this->document->setMetaData('twitter:url', $url);
		if ($description = $this->document->getMetaData('description'))
		{
			$this->document->setMetaData('twitter:description', $description);
		}

		if ($image = $this->document->getMetaData('image'))
		{
			$this->document->setMetaData('twitter:image', $image);
		}

		// Set meta open graph
		$this->document->setMetadata('og:type', 'website', 'property');
		$this->document->setMetaData('og:site_name', $sitename, 'property');
		$this->document->setMetaData('og:title', $title, 'property');
		$this->document->setMetaData('og:url', $url, 'property');

		if ($description)
		{
			$this->document->setMetaData('og:description', $description, 'property');
		}

		if ($image)
		{
			$this->document->setMetaData('og:image', $image, 'property');
		}
	}

	/**
	 * Method to check review double.
	 *
	 * @throws \Exception
	 *
	 * @since  1.1.0
	 */
	protected function checkReviewDoubles()
	{
		$app         = Factory::getApplication();
		$language    = $app->getLanguage()->getTag();
		$currentPath = Uri::getInstance()->toString(['path', 'query', 'fragment']);
		$reviewPath  = Route::_(RouteHelper::getReviewRoute($this->item->id, $this->item->item_id, $this->item->context, $language));

		if ($currentPath !== $reviewPath)
		{
			$app->redirect($reviewPath, 301);
		}
	}
}