<?php
/*
 * @package   pkg_radicalreviews
 * @version   1.1.0
 * @author    Delo Design
 * @copyright Copyright (c) 2023 Delo Design. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://delo-design.ru
 */

namespace Joomla\Component\RadicalReviews\Site\Model;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Filesystem\Folder;
use Joomla\CMS\Filesystem\Path;
use Joomla\CMS\MVC\Model\ListModel;
use Joomla\CMS\Router\Route;
use Joomla\Component\RadicalReviews\Administrator\Helper\EventHelper;
use Joomla\Component\RadicalReviews\Administrator\Helper\MediaHelper;
use Joomla\Component\RadicalReviews\Administrator\Traits\AdminModelTrait;
use Joomla\Component\RadicalReviews\Site\Helper\ReviewsHelper;
use Joomla\Component\RadicalReviews\Site\Helper\RouteHelper;
use Joomla\Database\QueryInterface;
use Joomla\Registry\Registry;
use Joomla\Utilities\ArrayHelper;

class ReviewsModel extends ListModel
{
	use AdminModelTrait;

	/**
	 * Model context string.
	 *
	 * @var  string
	 *
	 * @since  1.0.0
	 */
	protected $_context = 'com_radicalreviews.reviews';

	/**
	 * Method to auto-populate the model state.
	 *
	 * @param   string  $ordering   An optional ordering field.
	 * @param   string  $direction  An optional direction (asc|desc).
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	protected function populateState($ordering = null, $direction = null)
	{
		$app = Factory::getApplication('site');

		$object = ReviewsHelper::parseItem($app->input->getString('item'));

		// Set request states
		$this->setState('object.item_id', $object->item_id);
		$this->setState('object.context', $object->context);

		// Set params state
		$params = $app->getParams();
		$this->setState('params', $params);

		// Set published && debug state
		if ($app->input->getInt('debug', 0))
		{
			$this->setState('filter.published', array(0, 1));
			$this->setState('debug', 1);
		}
		else
		{
			$this->setState('filter.published', 1);
		}

		// List state information
		$ordering  = empty($ordering) ? 'r.created' : $ordering;
		$direction = empty($direction) ? 'desc' : $direction;

		parent::populateState($ordering, $direction);

		// Set ordering for query
		$this->setState('list.ordering', $ordering);
		$this->setState('list.direction', $direction);

		// Set limit & start for query
		$this->setState('list.limit', $params->get('reviews_limit', 10, 'uint'));
		$this->setState('list.start', $app->input->get('start', 0, 'uint'));
	}

	/**
	 * Method to get a store id based on model configuration state.
	 *
	 * @param   string  $id  A prefix for the store id.
	 *
	 * @return  string  A store id.
	 *
	 * @since  1.0.0
	 */
	protected function getStoreId($id = '')
	{
		$id .= ':' . $this->getState('object.item_id');
		$id .= ':' . $this->getState('object.context');
		$id .= ':' . serialize($this->getState('filter.published'));

		return parent::getStoreId($id);
	}

	/**
	 * Method to get a JDatabaseQuery object for retrieving the data set from a database.
	 *
	 * @return  QueryInterface  A JDatabaseQuery object to retrieve the data set.
	 *
	 * @since  1.0.0
	 */
	protected function getListQuery()
	{
		$db    = $this->getDatabase();
		$query = $db->getQuery(true)
			->select(array('r.*'))
			->from($db->quoteName('#__radicalreviews_reviews', 'r'));

		// Join over the users
		$query->select(array('u.name as author_name'))
			->leftJoin($db->quoteName('#__users', 'u') . ' ON u.id = r.created_by');

		// Filter by published state
		$published = $this->getState('filter.published');
		if (is_numeric($published))
		{
			$query->where('r.state = ' . (int) $published);
		}
		elseif (is_array($published))
		{
			$published = ArrayHelper::toInteger($published);
			$published = implode(',', $published);

			$query->where('r.state IN (' . $published . ')');
		}

		// Filter by item_id and context
		if ($this->getState('object.item_id') && $this->getState('object.context'))
		{
			$query->where('r.item_id = ' . $this->getState('object.item_id'));
			$query->where('r.context = ' . $db->quote($this->getState('object.context')));
		}

		// Group by
		$query->group(array('r.id'));

		// Add the list ordering clause
		$ordering  = $this->state->get('list.ordering', 'r.created');
		$direction = $this->state->get('list.direction', 'desc');
		$query->order($db->escape($ordering) . ' ' . $db->escape($direction));

		return $query;
	}

	/**
	 * Method to get an array of reviews data.
	 *
	 * @return  mixed  Reviews objects array on success, false on failure.
	 *
	 * @since  1.0.0
	 */
	public function getItems()
	{
		if ($items = parent::getItems())
		{
			foreach ($items as $item)
			{
				// Set author
				if (empty($item->author_name))
				{
					$item->author_name = $item->name;
				}

				// Set media
				$item->media = array();
				$item->image = '';
				$item->video = false;
				$folder      = 'images/reviews/' . $item->id;
				$path        = Path::clean(JPATH_ROOT . '/' . $folder);

				// Trigger event
				$object       = EventHelper::getObject($this, $item->item_id, $item->context);
				$item->link   = Route::link('site', RouteHelper::getReviewRoute($item->id, $item->item_id, $item->context), false);
				$item->object = $object;
				$item->source = !empty($item->source) ? new Registry($item->source) : false;

				if (Folder::exists($path))
				{
					$files = Folder::files($path, '^((?!poster).)*$');

					foreach ($files as $file)
					{
						$item->media[] = $folder . '/' . $file;
					}
				}

				if (!empty($item->media))
				{
					if (MediaHelper::getInstance()->checkMedia(JPATH_ROOT . '/' . $item->media[0]) === 'image')
					{
						$item->image = $item->media[0];
					}
					else
					{
						$item->video = true;
					}

					// Check image
					if (!$item->image && isset($item->object->image))
					{
						$item->image = $item->object->image;
					}
				}
			}
		}

		return $items;
	}
}