<?php
/*
 * @package   pkg_radicalreviews
 * @version   1.1.0
 * @author    Delo Design
 * @copyright Copyright (c) 2023 Delo Design. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://delo-design.ru
 */

namespace Joomla\Component\RadicalReviews\Administrator\Model;

defined('_JEXEC') or die;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Filesystem\Folder;
use Joomla\CMS\Filesystem\Path;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Form\FormHelper;
use Joomla\CMS\MVC\Model\AdminModel;
use Joomla\CMS\Object\CMSObject;
use Joomla\CMS\Table\Table;
use Joomla\CMS\User\UserFactoryInterface;
use Joomla\Component\RadicalReviews\Administrator\Helper\EventHelper;
use Joomla\Component\RadicalReviews\Administrator\Helper\MediaHelper;
use Joomla\Component\RadicalReviews\Administrator\Traits\AdminModelTrait;
use Joomla\Registry\Registry;

class ReviewModel extends AdminModel
{
	use AdminModelTrait;

	/**
	 * Method to get a single record.
	 *
	 * @param   integer  $pk  The id of the primary key.
	 *
	 * @return  CMSObject|boolean  Object on success, false on failure.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function getItem($pk = null)
	{
		if ($item = parent::getItem($pk))
		{
			// Set defaults
			$item->media       = array();
			$item->image       = false;
			$item->author_name = '';

			if (!empty($item->id))
			{
				// Set media
				$folder = 'images/reviews/' . $item->id;
				$path   = Path::clean(JPATH_ROOT . '/' . $folder);
				if (Folder::exists($path))
				{
					$files = Folder::files($path, '^((?!poster).)*$');

					foreach ($files as $file)
					{
						$item->media[] = $folder . '/' . $file;
					}
				}

				if (!empty($item->media))
				{
					if (MediaHelper::getInstance()->checkMedia(JPATH_ROOT . '/' . $item->media[0]) === 'image')
					{
						$item->image = $item->media[0];
					}
//					else
//					{
//						$item->image = basename($item->media[0]) . '_poster.jpg';
//					}
				}

				// Set object
				if (!empty($item->item_id))
				{
					$item->object = $this->getObject($item->item_id, $item->context);

					if (!$item->image && isset($item->object->image))
					{
						$item->image = $item->object->image;
					}

					$item->author_name = $item->name;

					// Set Author name
					if (!empty($item->created_by))
					{
						$user = Factory::getContainer()->get(UserFactoryInterface::class)->loadUserById($item->created_by);

						if ($user)
						{
							$item->author_name = $user->name;
						}
					}
				}
			}

			// Set source
			$item->source = new Registry($item->source);
		}

		return $item;
	}

	/**
	 * Returns a Table object, always creating it.
	 *
	 * @param   string  $name     The table type to instantiate
	 * @param   string  $prefix   A prefix for the table class name.
	 * @param   array   $options  Configuration array for model.
	 *
	 * @return  Table  A database object.
	 *
	 * @since  1.0.0
	 */
	public function getTable($name = 'Reviews', $prefix = 'Administrator', $options = array())
	{
		return parent::getTable($name, $prefix, $options);
	}

	/**
	 * Abstract method for getting the form from the model.
	 *
	 * @param   array    $data      Data for the form.
	 * @param   boolean  $loadData  True if the form is to load its own data (default case), false if not.
	 *
	 * @return  Form|boolean  A Form object on success, false on failure.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function getForm($data = array(), $loadData = true)
	{
		$form = $this->loadForm('com_radicalreviews.review', 'review', array('control' => 'jform', 'load_data' => $loadData));
		if (empty($form))
		{
			return false;
		}

		// Get review id
		$id = (int) $this->getState('review.id', Factory::getApplication()->input->get('id', 0));

		// Modify the form based on Edit State access controls
		if ($id != 0 && !Factory::getApplication()->getIdentity()->authorise('core.edit.state', 'com_radicalreviews.review.' . $id))
		{
			$form->setFieldAttribute('state', 'disabled', 'true');
			$form->setFieldAttribute('state', 'filter', 'unset');
		}

		// Admin form
		if (Factory::getApplication()->isClient('administrator'))
		{
			if ($context = $form->getValue('context'))
			{
				$form->setFieldAttribute('context', 'readonly', 'true');

				$contexts = EventHelper::getContexts($this);

				if (isset($context))
				{
					// Load component language
					list($component,) = explode('.', $context, 2);
					Factory::getApplication()->getLanguage()->load($component, JPATH_ADMINISTRATOR);

					// Set field namespace
					FormHelper::addFieldPrefix($contexts[$context]['prefix']);

					// Change field type
					$form->setFieldAttribute('item_id', 'type', $contexts[$context]['field']);
					$form->setFieldAttribute('item_id', 'new', false);
					$form->setFieldAttribute('item_id', 'edit', false);
				}
			}
			else
			{
				$fields = ['item_id', 'media', 'rating', 'text', 'source', 'created_by', 'name', 'email'];

				foreach ($fields as $field)
				{
					$form->setFieldAttribute($field, 'readonly', 'true');
				}
			}
		}

		return $form;
	}

	/**
	 * Method to get the data that should be injected in the form.
	 *
	 * @return  mixed  The data for the form.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	protected function loadFormData()
	{
		$data = Factory::getApplication()->getUserState('com_radicalreviews.edit.review.data', array());
		if (empty($data))
		{
			$data = $this->getItem();
		}

		$this->preprocessData('com_radicalreviews.review', $data);

		return $data;
	}

	/**
	 * Method to save the form data.
	 *
	 * @param   array  $data  The form data.
	 *
	 * @return  boolean  True on success.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function save($data)
	{
		$pk      = (!empty($data['id'])) ? $data['id'] : (int) $this->getState($this->getName() . '.id');
		$table   = $this->getTable();
		$isNew   = true;
		$app     = Factory::getApplication();
		$nowDate = Factory::getDate()->toSql();
		$userId  = $app->getIdentity()->id;

		// Load the row if saving an existing review
		if ($pk > 0)
		{
			$table->load($pk);
			$isNew = false;
		}

		if ($isNew)
		{
			// Set created field data
			if (empty($data['created']))
			{
				$data['created'] = $nowDate;
			}

			// Set created_by field data
			if (empty($data['created_by']) && isset($data['name']) && empty($data['name']))
			{
				$data['created_by'] = $userId;
			}
		}

		// Prepare source field data
		if (isset($data['source']))
		{
			$data['source'] = (new Registry($data['source']))->toString();
		}

		// Prepare source field data
		if (isset($data['plugins']))
		{
			$data['plugins'] = (new Registry($data['plugins']))->toString();
		}

		// Set modified
		$data['modified']    = $nowDate;
		$data['modified_by'] = $userId;

		if (parent::save($data))
		{
			$id = $this->getState($this->getName() . '.id');
			if (empty($id)) $id = $table->id;

			// Upload media
			if (!empty($data['media']))
			{
				// Max media
				$data['media'] = array_slice($data['media'], 0, ComponentHelper::getParams($this->option)->get('media_count', 3));

				// Upload media
				MediaHelper::getInstance()->uploadMedia($id, $data['media']);
			}

			// Minus content rating
			if (empty($data['minus_rating']) && $isNew)
			{
				$db    = Factory::getContainer()->get('DatabaseDriver');
				$query = $db->getQuery(true)
					->select('*')
					->from($db->quoteName('#__radicalreviews_rating'))
					->where($db->quoteName('item_id') . ' = ' . (int) $data['item_id'])
					->where($db->quoteName('context') . ' = ' . (int) $data['context']);
				if ($update = $db->setQuery($query)->loadObject())
				{
					$update->votes    = (int) $update->votes + 1;
					$update->stars    = (int) $update->stars + (int) $data['rating'];
					$update->modified = Factory::getDate()->toSql();

					$db->updateObject('#__radicalreviews_rating', $update, 'id');
				}
				else
				{
					$insert           = new \stdClass();
					$insert->item_id  = $data['item_id'];
					$insert->context  = $data['context'];
					$insert->stars    = $data['rating'];
					$insert->votes    = 1;
					$insert->modified = Factory::getDate()->toSql();

					$db->insertObject('#__radicalreviews_rating', $insert);
				}
			}

			return $id;
		}

		return false;
	}

	/**
	 * Check if file is media by mime type.
	 *
	 * @param   string  $media  Full path to image.
	 *
	 * @return  true|false  True on success, false on failure.
	 *
	 * @since  1.0.0
	 */
	public function checkMedia($media = null)
	{
		return MediaHelper::getInstance()->checkMedia($media);
	}

	/**
	 * Clean the cache.
	 *
	 * @param   string   $group      The cache group.
	 * @param   integer  $client_id  The ID of the client.
	 *
	 * @since   1.0.0
	 */
	protected function cleanCache($group = null, $client_id = 0)
	{
		foreach (array($group, 'com_radicalreviews') as $group) parent::cleanCache($group, $client_id);
	}

	/**
	 * Method to delete one or more records.
	 *
	 * @param   array  &$pks  An array of record primary keys.
	 *
	 * @return  boolean  True if successful, false if an error occurs.
	 *
	 * @since   1.1.0
	 */
	public function delete(&$pks)
	{
		if ($result = parent::delete($pks))
		{
			foreach ($pks as $pk)
			{
				$folder = Path::clean(JPATH_ROOT . '/images/reviews/' . $pk);
				if (Folder::exists($folder)) Folder::delete($folder);
			}
		}

		return $result;
	}
}