<?php
/*
 * @package   pkg_radicalreviews
 * @version   1.1.0
 * @author    Delo Design
 * @copyright Copyright (c) 2023 Delo Design. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://delo-design.ru
 */

namespace Joomla\Component\RadicalReviews\Site\View\Reviews;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Multilanguage;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\View\HtmlView as BaseHtmlView;
use Joomla\CMS\Pagination\Pagination;
use Joomla\CMS\Uri\Uri;
use Joomla\Component\RadicalReviews\Site\Helper\ReviewsHelper;
use Joomla\Registry\Registry;

class HtmlView extends BaseHtmlView
{
	/**
	 * Model state variables.
	 *
	 * @var  \Joomla\CMS\Object\CMSObject
	 *
	 * @since   1.0.0
	 */
	protected $state;

	/**
	 * Application params.
	 *
	 * @var  Registry
	 *
	 * @since   1.0.0
	 */
	public $params;

	/**
	 * Object.
	 *
	 * @var  object|false
	 *
	 * @since  1.0.0
	 */
	protected $object;

	/**
	 * Reviews array.
	 *
	 * @var  array
	 *
	 * @since   1.0.0
	 */
	protected $items;

	/**
	 * Pagination object.
	 *
	 * @var  Pagination
	 *
	 * @since  1.0.0
	 */
	protected $pagination;

	/**
	 * Page class suffix from params.
	 *
	 * @var  string
	 *
	 * @since  1.0.0
	 */
	public $pageclass_sfx;

	/**
	 * Display the view.
	 *
	 * @param   string  $tpl  The name of the template file to parse.
	 *
	 * @return  mixed  A string if successful, otherwise an Error object.
	 *
	 * @throws  \Exception
	 *
	 * @since   1.0.0
	 */
	public function display($tpl = null)
	{
		$this->state      = $this->get('State');
		$this->params     = $this->state->get('params');
		$this->items      = $this->get('Items');
		$this->object     = $this->get('Object');
		$this->pagination = $this->get('Pagination');

		// Check for errors
		if (count($errors = $this->get('Errors')))
		{
			throw new \Exception(implode('\n', $errors), 500);
		}

		// Check get params
		if (!$this->object || $this->object->state !== 1 || !$this->state->get('object.context') || !$this->state->get('object.item_id'))
		{
			throw new \Exception(Text::_('COM_RADICALREVIEWS_ERROR_REVIEWS_NOT_FOUND'), 404);
		}

		// Check multilanguage redirect
		ReviewsHelper::checkMultilanguage($this->object);

		// Check reviewed state
		$key = 'new_object_review_' . $this->state->get('object.context') . '_' . $this->state->get('object.item_id');
		$this->reviewed = Factory::getApplication()->getUserState($key);

		// Check doubled
		$this->checkPaginationDoubles();

		// Load plugin language
		$app = Factory::getApplication();
		$app->getLanguage()->load('plg_radicalreviews_' . explode('.' . $this->object->context, 1)[0], JPATH_ADMINISTRATOR);

		// Prepare the document
		if (!empty((array) $this->object))
		{

			$this->_prepareDocument();
		}

		parent::display($tpl);
	}

	/**
	 * Prepare the document.
	 *
	 * @throws  \Exception
	 *
	 * @since   1.0.0
	 */
	protected function _prepareDocument()
	{
		$app    = Factory::getApplication();
		$object = $this->object;

		// Change breadcrumbs
		$pathway = $app->getPathway();

		$pathwayObject       = new \stdClass();
		$pathwayObject->name = $this->object->title;
		$pathwayObject->link = $this->object->link;
		$pathwayItems[]      = $pathwayObject;

		$pathwayItem       = new \stdClass();
		$pathwayItem->name = Text::_('COM_RADICALREVIEWS_REVIEWS');
		$pathwayItem->link = '';
		$pathwayItems[]    = $pathwayItem;

		$pathway->setPathway($pathwayItems);

		// Set meta title
		$title    = Text::sprintf('COM_RADICALREVIEWS_REVIEWS_TITLE', $object->title);
		$sitename = $app->get('sitename');

		if ($app->get('sitename_pagetitles', 0) == 1)
		{
			$title = Text::sprintf('JPAGETITLE', $sitename, $title);
		}
		elseif ($app->get('sitename_pagetitles', 0) == 2)
		{
			$title = Text::sprintf('JPAGETITLE', $title, $sitename);
		}
		$this->document->setTitle($title);

		// Set description
		$this->document->setDescription(Text::sprintf('COM_RADICALREVIEWS_REVIEWS_DESCRIPTION', $object->title));

		// Set pagination
		if ($this->pagination)
		{
			$title = Text::sprintf('COM_RADICALREVIEWS_PAGINATION_SEO_TITLE', $title,
				Text::sprintf('COM_RADICALREVIEWS_PAGINATION', $this->pagination->pagesCurrent));
			$this->document->setTitle($title);
			$this->document->setDescription('');
		}

		// Set meta image
		$this->document->setMetadata('image', Uri::root() . $object->image);

		// Set meta robots
		if ($this->state->get('debug', 0))
		{
			$this->document->setMetadata('robots', 'noindex');
		}
		elseif ($this->params->get('robots'))
		{
			$this->document->setMetadata('robots', $this->params->get('robots'));
		}

		// Set meta url
		$url = Uri::getInstance()->toString(array('scheme', 'host', 'port')) . $object->link;
		$this->document->setMetaData('url', $url);

		// Set meta twitter
		$this->document->setMetaData('twitter:card', 'summary_large_image');
		$this->document->setMetaData('twitter:site', $sitename);
		$this->document->setMetaData('twitter:creator', $sitename);
		$this->document->setMetaData('twitter:title', $title);
		$this->document->setMetaData('twitter:url', $url);

		if ($description = $this->document->getMetaData('description'))
		{
			$this->document->setMetaData('twitter:description', $description);
		}

		if ($image = $this->document->getMetaData('image'))
		{
			$this->document->setMetaData('twitter:image', $image);
		}

		// Set meta open graph
		$this->document->setMetadata('og:type', 'website', 'property');
		$this->document->setMetaData('og:site_name', $sitename, 'property');
		$this->document->setMetaData('og:title', $title, 'property');
		$this->document->setMetaData('og:url', $url, 'property');

		if ($description)
		{
			$this->document->setMetaData('og:description', $description, 'property');
		}

		if ($image)
		{
			$this->document->setMetaData('og:image', $image, 'property');
		}
	}

	/**
	 * Method to check pagination start and redirect if needed.
	 *
	 * @throws \Exception
	 *
	 * @since  1.0.0
	 */
	protected function checkPaginationDoubles()
	{
		$app   = Factory::getApplication();
		$start = $app->input->getInt('start');

		if ($this->pagination && $start > 0)
		{
			$redirect = false;
			$uri      = Uri::getInstance();

			if ($this->pagination->limit === 0)
			{
				$uri->delVar('start');
				$redirect = true;
			}
			elseif (($app->input->getInt('start') / $this->pagination->limit) + 1 !== $this->pagination->pagesCurrent)
			{
				$uri->setVar('start', ($this->pagination->pagesCurrent - 1) * $this->pagination->limit);
				$redirect = true;
			}

			if ($redirect)
			{
				$app->redirect($uri->toString(['path', 'query', 'fragment']), 301);
			}
		}
	}
}