<?php
/*
 * @package   pkg_radicalreviews
 * @version   1.1.0
 * @author    Delo Design
 * @copyright Copyright (c) 2023 Delo Design. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://delo-design.ru
 */

namespace Joomla\Component\RadicalReviews\Administrator\Model;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\MVC\Model\ListModel;
use Joomla\Component\RadicalReviews\Administrator\Traits\AdminModelTrait;
use Joomla\Utilities\ArrayHelper;

class RatingModel extends ListModel
{
	use AdminModelTrait;

	/**
	 * Constructor.
	 *
	 * @param   array  $config  An optional associative array of configuration settings.
	 *
	 * @since  1.0.0
	 */
	public function __construct($config = array())
	{
		// Add the ordering filtering fields whitelist
		if (empty($config['filter_fields']))
		{
			$config['filter_fields'] = array(
				'id', 'r.id',
				'item_id', 'r.item_id',
				'context', 'r.context',
				'votes', 'r.votes', 'rating',
				'modified', 'r.modified',
			);
		}

		parent::__construct($config);
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * @param   string  $ordering   An optional ordering field.
	 * @param   string  $direction  An optional direction (asc|desc).
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	protected function populateState($ordering = null, $direction = null)
	{
		// Set search filter state
		$search = $this->getUserStateFromRequest($this->context . '.filter.search', 'filter_search');
		$this->setState('filter.search', $search);

		// Set item_id filter state
		$item_id = $this->getUserStateFromRequest($this->context . '.filter.item_id', 'filter_item_id', '');
		$this->setState('filter.item_id', $item_id);

		// Set context filter state
		$context = $this->getUserStateFromRequest($this->context . '.filter.context', 'filter_context', '');
		$this->setState('filter.context', $context);

		// List state information
		$ordering  = empty($ordering) ? 'r.modified' : $ordering;
		$direction = empty($direction) ? 'desc' : $direction;

		parent::populateState($ordering, $direction);
	}

	/**
	 * Method to get a store id based on model configuration state.
	 *
	 * @param   string  $id  A prefix for the store id.
	 *
	 * @return  string  A store id.
	 *
	 * @since  1.0.0
	 */
	protected function getStoreId($id = '')
	{
		$id .= ':' . $this->getState('filter.search');
		$id .= ':' . $this->getState('filter.context');
		$id .= ':' . $this->getState('filter.item_id');

		return parent::getStoreId($id);
	}

	/**
	 * Method to get a DatabaseQuery object for retrieving the data set from a database.
	 *
	 * @return  \Joomla\Database\QueryInterface  A QueryInterface object to retrieve the data set.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	protected function getListQuery()
	{
		$db    = $this->getDatabase();
		$query = $db->getQuery(true)
			->select(array('r.*', 'r.stars/r.votes as rating'))
			->from($db->quoteName('#__radicalreviews_rating', 'r'));

		// Filter by item_id state
		$item_id = $this->getState('filter.item_id');
		if (is_numeric($item_id))
		{
			$query->where('r.item_id = ' . (int) $item_id);
		}

		// Filter by context state
		$context = $this->getState('filter.context');
		if ($context)
		{
			$query->where('r.context = ' . $context);
		}

		// Group by
		$query->group(array('r.id'));

		// Add the list ordering clause
		$ordering  = $this->state->get('list.ordering', 'r.modified');
		$direction = $this->state->get('list.direction', 'desc');
		$query->order($db->escape($ordering) . ' ' . $db->escape($direction));

		return $query;
	}

	/**
	 * Method to get an array of projects data.
	 *
	 * @return  mixed  Projects objects array on success, false on failure.
	 *
	 * @since  1.0.0
	 */
	public function getItems()
	{
		if ($items = parent::getItems())
		{
			foreach ($items as $item)
			{
				// Set rating
				$item->rating = ceil($item->rating);

				// Set object
				$item->object = $this->getObject($item->item_id, $item->context);
			}
		}

		return $items;
	}

	/**
	 * Method to delete one or more records.
	 *
	 * @param   array  &$pks  An array of record primary keys.
	 *
	 * @return  boolean  True if successful, false if an error occurs.
	 *
	 * @since   1.6
	 */
	public function delete(&$pks)
	{
		$pks   = ArrayHelper::toInteger((array) $pks);
		$table = $this->getTable();

		// Iterate the items to delete each one.
		foreach ($pks as $i => $pk)
		{
			if ($table->load($pk))
			{
				if (Factory::getApplication()->getIdentity()->authorise('core.delete', $this->option))
				{
					if (!$table->delete($pk))
					{
						$this->setError($table->getError());

						return false;
					}
				}
				else
				{
					// Prune items that you can't change.
					unset($pks[$i]);
					$error = $this->getError();

					if ($error)
					{
						Log::add($error, Log::WARNING, 'jerror');

						return false;
					}
					else
					{
						Log::add(Text::_('JLIB_APPLICATION_ERROR_DELETE_NOT_PERMITTED'), Log::WARNING, 'jerror');

						return false;
					}
				}
			}
			else
			{
				$this->setError($table->getError());

				return false;
			}
		}

		 // Clear the component's cache
        $this->cleanCache();

        return true;
	}
}