<?php
/*
 * @package   pkg_radicalreviews
 * @version   1.1.0
 * @author    Delo Design
 * @copyright Copyright (c) 2023 Delo Design. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://delo-design.ru
 */

namespace Joomla\Component\RadicalReviews\Administrator\Model;

defined('_JEXEC') or die;

use Joomla\CMS\MVC\Model\ListModel;
use Joomla\Component\RadicalReviews\Administrator\Traits\AdminModelTrait;

class ReviewsModel extends ListModel
{
	use AdminModelTrait;

	/**
	 * Constructor.
	 *
	 * @param   array  $config  An optional associative array of configuration settings.
	 *
	 * @since  1.0.0
	 */
	public function __construct($config = array())
	{
		// Add the ordering filtering fields whitelist
		if (empty($config['filter_fields']))
		{
			$config['filter_fields'] = array(
				'id', 'r.id',
				'published', 'state', 'r.state',
				'created', 'r.created',
				'rating', 'r.rating',
				'item_id', 'r.item_id', 'object_title'
			);
		}

		parent::__construct($config);
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * @param   string  $ordering   An optional ordering field.
	 * @param   string  $direction  An optional direction (asc|desc).
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	protected function populateState($ordering = null, $direction = null)
	{
		// Set search filter state
		$search = $this->getUserStateFromRequest($this->context . '.filter.search', 'filter_search');
		$this->setState('filter.search', $search);

		// Set published filter state
		$published = $this->getUserStateFromRequest($this->context . '.filter.published', 'filter_published', '');
		$this->setState('filter.published', $published);

		// Set item_id filter state
		$item_id = $this->getUserStateFromRequest($this->context . '.filter.item_id', 'filter_item_id', '');
		$this->setState('filter.item_id', $item_id);

		// Set context filter state
		$context = $this->getUserStateFromRequest($this->context . '.filter.context', 'filter_context', '');
		$this->setState('filter.context', $context);

		// List state information
		$ordering  = empty($ordering) ? 'r.created' : $ordering;
		$direction = empty($direction) ? 'desc' : $direction;

		parent::populateState($ordering, $direction);
	}

	/**
	 * Method to get a store id based on model configuration state.
	 *
	 * @param   string  $id  A prefix for the store id.
	 *
	 * @return  string  A store id.
	 *
	 * @since  1.0.0
	 */
	protected function getStoreId($id = '')
	{
		$id .= ':' . $this->getState('filter.search');
		$id .= ':' . $this->getState('filter.published');
		$id .= ':' . $this->getState('filter.context');
		$id .= ':' . $this->getState('filter.item_id');

		return parent::getStoreId($id);
	}

	/**
	 * Method to get a DatabaseQuery object for retrieving the data set from a database.
	 *
	 * @return  \Joomla\Database\QueryInterface  A QueryInterface object to retrieve the data set.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	protected function getListQuery()
	{
		$db    = $this->getDatabase();
		$query = $db->getQuery(true)
			->select(array('r.*'))
			->from($db->quoteName('#__radicalreviews_reviews', 'r'));

		// Join over the users
		$query->select(array('u.name as author_name'))
			->leftJoin($db->quoteName('#__users', 'u') . ' ON u.id = r.created_by');

		// Filter by published state
		$published = $this->getState('filter.published');
		if (is_numeric($published))
		{
			$query->where('r.state = ' . (int) $published);
		}
		elseif ($published === '')
		{
			$query->where('(r.state = 0 OR r.state = 1)');
		}

		// Filter by item_id state
		$item_id = $this->getState('filter.item_id');
		if (is_numeric($item_id))
		{
			$query->where('r.item_id = ' . (int) $item_id);
		}

		// Filter by search
		$search = $this->getState('filter.search');
		if (!empty($search))
		{
			if (stripos($search, 'id:') === 0)
			{
				$query->where('r.id = ' . (int) substr($search, 3));
			}
			else if (stripos($search, 'item_id:') === 0)
			{
				$query->where('r.item_id = ' . (int) substr($search, 8));
			}
			else
			{
				$sql     = array();
				$columns = array('r.text', 'r.name', 'u.name');
				foreach ($columns as $column)
				{
					$sql[] = $db->quoteName($column) . ' LIKE '
						. $db->quote('%' . str_replace(' ', '%', $db->escape(trim($search), true) . '%'));
				}
				$query->where('(' . implode(' OR ', $sql) . ')');
			}
		}

		// Group by
		$query->group(array('r.id'));

		// Add the list ordering clause
		$ordering  = $this->state->get('list.ordering', 'r.created');
		$direction = $this->state->get('list.direction', 'desc');
		$query->order($db->escape($ordering) . ' ' . $db->escape($direction));

		return $query;
	}

	/**
	 * Method to get an array of projects data.
	 *
	 * @return  mixed  Projects objects array on success, false on failure.
	 *
	 * @since  1.0.0
	 */
	public function getItems()
	{
		if ($items = parent::getItems())
		{
			foreach ($items as $item)
			{
				// Set text
				$item->text = nl2br($item->text);

				// Set object
				$item->object = $this->getObject($item->item_id, $item->context);

				// Set author
				if (empty($item->author_name))
				{
					$item->author_name = $item->name;
				}
			}
		}

		return $items;
	}
}