<?php
/*
 * @package   pkg_radicalreviews
 * @version   1.1.0
 * @author    Delo Design
 * @copyright Copyright (c) 2023 Delo Design. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://delo-design.ru
 */

namespace Joomla\Component\RadicalReviews\Administrator\Helper;

defined('_JEXEC') or die;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Filesystem\File;
use Joomla\CMS\Filesystem\Folder;
use Joomla\CMS\Filesystem\Path;
use Joomla\CMS\Language\Text;

class MediaHelper
{
	/**
	 * Stores the singleton instances of various toolbar.
	 *
	 * @var    MediaHelper
	 *
	 * @since  1.1.0
	 */
	protected static $instance = array();

	/**
	 * Accepted media file types.
	 *
	 * @var    array
	 *
	 * @since  1.1.0
	 */
	protected $imageTypes = array(
		'image/jpeg',
		'image/png',
		'image/gif',
		'image/bmp'
	);

	/**
	 * Accepted media file types.
	 *
	 * @var    array
	 *
	 * @since  1.1.0
	 */
	protected $videoTypes = array(
		'video/mpeg',
		'video/mp4'
//		'video/ogg',
//		'video/quicktime',
//		'video/webm',
//		'video/x-ms-wmv',
//		'video/x-flv',
//		'video/x-msvideo',
//		'video/3gpp',
//		'video/3gpp2'
	);

	/**
	 * Returns the global MediaHelper object, only creating it if it doesn't already exist
	 *
	 * @return  MediaHelper  The MediaHelper object.
	 *
	 * @since       1.1.0
	 */
	public static function getInstance()
	{
		if (empty(self::$instance))
		{
			self::$instance = new MediaHelper();
		}

		return self::$instance;
	}

	/**
	 * Check if file is image or video by mime type.
	 *
	 * @param   string  $media  Full path to media.
	 *
	 * @return  string|false  String on success, false on failure.
	 *
	 * @since  1.0.0
	 */
	public function checkMedia($media = null)
	{
		$media = Path::clean($media);
		if (function_exists('finfo_open'))
		{
			$finfo    = finfo_open(FILEINFO_MIME_TYPE);
			$mimetype = finfo_file($finfo, $media);
			finfo_close($finfo);
		}
		else
		{
			$mimetype = mime_content_type($media);
		}

		if (in_array($mimetype, $this->imageTypes))
		{
			return 'image';
		}
		else if (in_array($mimetype, $this->videoTypes))
		{
			return 'video';
		}

		return false;
	}

	/**
	 * Check media size.
	 *
	 * @param   string  $media  Upload media.
	 *
	 * @return  string|true  True on success, false on failure.
	 *
	 * @since  1.1.0
	 */
	public function checkError($media = null)
	{
		// Make sure that file uploads are enabled in php.
		if (!(bool) ini_get('file_uploads'))
		{
			return Text::_('COM_RADICALREVIEWS_MEDIA_ERROR_UPLOAD_FILE');
		}

		// Check user file
		if (!is_array($media))
		{
			return Text::_('COM_RADICALREVIEWS_MEDIA_ERROR_NO_FILE_SELECTED');
		}

		// Check tmp directory
		if ($media['error'] && ($media['error'] == UPLOAD_ERR_NO_TMP_DIR))
		{
			return Text::_('COM_RADICALREVIEWS_MEDIA_ERROR_PHP_UPLOAD_NOT_SET');
		}

		// Check max upload size in php.ini
		if ($media['error'] && ($media['error'] == UPLOAD_ERR_INI_SIZE))
		{
			return Text::_('COM_RADICALREVIEWS_MEDIA_ERROR_SMALL_UPLOAD_SIZE');
		}

		// Check other errors.
		if ($media['error'] || $media['size'] < 1)
		{
			return Text::_('COM_RADICALREVIEWS_MEDIA_ERROR_UPLOAD_ERROR');
		}

		// Check max size from config
		$maxFileSize = ComponentHelper::getParams('com_radicalreviews')->get('media_size', 2);

		if ($media['size'] > $maxFileSize * 1024 * 1024)
		{
			return Text::sprintf('COM_RADICALREVIEWS_MEDIA_ERROR_MAX_UPLOAD_SIZE', $maxFileSize);
		}

		return false;
	}

	/**
	 * Method to generate random image name.
	 *
	 * @param   int  $length  Name length.
	 *
	 * @return  string  Image name.
	 *
	 * @since  1.0.0
	 */
	public function generateMediaName($length = 11)
	{
		$name  = '';
		$chars = array('a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'r', 's',
			't', 'u', 'v', 'x', 'y', 'z', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O',
			'P', 'R', 'S', 'T', 'U', 'V', 'X', 'Y', 'Z', 0, 1, 2, 3, 4, 5, 6, 7, 8, 9);
		for ($i = 0; $i < $length; $i++)
		{
			$key  = rand(0, count($chars) - 1);
			$name .= $chars[$key];
		}

		return $name;
	}

	/**
	 * Method to upload image.
	 *
	 * @param   int    $id     Review id.
	 * @param   array  $media  Array of medias from global $_FILES array.
	 *
	 * @return  bool
	 *
	 * @since  1.1.0
	 */
	public function uploadMedia(int $id, array $media)
	{
		$folder = Path::clean(JPATH_ROOT . '/images/reviews/' . $id);

		if (!Folder::exists($folder))
		{
			Folder::create($folder);
		}

		$names = array();

		foreach ($media as $file)
		{
			if ($this->checkMedia($file['tmp_name']))
			{
				$name = $this->generateMediaName();
				while (in_array($name, $names))
				{
					$name = $this->generateMediaName();
				}
				$filename = $name . '.' . File::getExt($file['name']);
				$dest     = Path::clean($folder . '/' . $filename);
				File::upload($file['tmp_name'], $dest, false, true);
			}
		}

		return true;
	}
}
