<?php
/*
 * @package   pkg_radicalreviews
 * @version   1.1.0
 * @author    Delo Design
 * @copyright Copyright (c) 2023 Delo Design. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://delo-design.ru
 */

namespace Joomla\Component\RadicalReviews\Site\Controller;

defined('_JEXEC') or die;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Filter\InputFilter;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\LayoutHelper;
use Joomla\CMS\MVC\Controller\FormController;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\CMS\Response\JsonResponse;
use Joomla\Component\RadicalReviews\Administrator\Helper\EventHelper;
use Joomla\Component\RadicalReviews\Administrator\Helper\MediaHelper;
use Joomla\Component\RadicalReviews\Administrator\Traits\JsonControllerTrait;
use Joomla\Component\RadicalReviews\Site\Model\FormModel;

class ReviewController extends FormController
{
	use JsonControllerTrait;

	/**
	 * The prefix to use with controller messages.
	 *
	 * @var  string
	 *
	 * @since  1.0.0
	 */
	protected $text_prefix = 'COM_RADICALREVIEWS_REVIEW';

	/**
	 * Method to save a record.
	 *
	 * @param   string  $key     The name of the primary key of the URL variable.
	 * @param   string  $urlVar  The name of the URL variable if different from the primary key (sometimes required to avoid router collisions).
	 *
	 * @return  boolean  True if successful, false otherwise.
	 *
	 * @throws  \Exception
	 *
	 * @since   1.0.0
	 */
	public function save($key = null, $urlVar = null)
	{
		return false;
	}

	/**
	 * Method to check if you can edit an existing record.
	 *
	 * Extended classes can override this if necessary.
	 *
	 * @param   array   $data  An array of input data.
	 * @param   string  $key   The name of the key for the primary key; default is id.
	 *
	 * @return  boolean
	 *
	 * @since   1.0.0
	 */
	protected function allowEdit($data = array(), $key = 'id')
	{
		return false;
	}

	/**
	 * Method to cancel an edit.
	 *
	 * @param   string  $key  The name of the primary key of the URL variable.
	 *
	 * @return  boolean  True if access level checks pass, false otherwise.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function cancel($key = null)
	{
		if ($result = parent::cancel($key))
		{
			$this->setRedirect($this->getModel()->getLink($this->input->getInt('item_id'), $this->input->getString('context'), 'reviews'));
		}

		return $result;
	}

	/**
	 * Method to check if you can add a new record.
	 *
	 * @param   array  $data  An array of input data.
	 *
	 * @return  boolean True on success, False on failure.
	 *
	 * @throws \Exception
	 *
	 * @since   1.0.0
	 */
	public function allowAdd($data = array())
	{
		return parent::allowAdd($data);
	}

	/**
	 * Method to validate media.
	 *
	 * @return boolean True on success, False on failure.
	 *
	 * @throws \Exception
	 *
	 * @since  1.0.0
	 */
	public function checkMedia()
	{
		$medias = $this->input->files->get('media', array(), 'array');
		$media  = (!empty($medias[0])) ? $medias[0] : false;

		// Check media
		$result  = false;
		$message = Text::_('COM_RADICALREVIEWS_ERROR_MEDIA_NOT_FOUND');

		if ($media)
		{
			$message = MediaHelper::getInstance()->checkError($media);

			if (!$message)
			{
				$message = ($result = MediaHelper::getInstance()->checkMedia($media['tmp_name'])) ? ''
				: Text::_('COM_RADICALREVIEWS_ERROR_MEDIA_FORMAT');
			}
		}

		header('Content-Type: application/json');
		echo new JsonResponse($result, $message, (!$result));
		Factory::getApplication()->close(($result) ? 200 : 404);

		return $result;
	}

	/**
	 * Method to add product to cart.
	 *
	 * @return  bool True on success, False on failure.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function addReview()
	{
		// Get review data
		if (!$data = $this->getReviewData())
		{
			return $this->setJsonResponse();
		}

		/* @var FormModel $model */
		$model = $this->getModel();

		// Save review
		if (!$reviewId = $model->save($data))
		{
			$this->message = [];
			foreach ($model->getErrors() as $error)
			{
				$this->message[] = ($error instanceof \Exception) ? $error->getMessage() : $error;
			}
			$this->code = 500;

			return $this->setJsonResponse();
		}

		// Save the data in the session
		Factory::getApplication()->setUserState('new_object_rating_' . $data['context'] . '_' . $data['item_id'], $data['rating']);
		Factory::getApplication()->setUserState('new_object_review_' . $data['context'] . '_' . $data['item_id'], true);

		// Get review
		if (!$review = $model->getItem($reviewId))
		{
			$this->message = [];
			foreach ($model->getErrors() as $error)
			{
				$this->message[] = ($error instanceof \Exception) ? $error->getMessage() : $error;
			}
			$this->code = 500;

			return $this->setJsonResponse();
		}

		// After save event
		EventHelper::afterSave($this, $review, $data);

		// Return layout
		$layout = $review->state ? 'review' : 'unpublished';
		$html   = LayoutHelper::render('components.radicalreviews.review.' . $layout, ['item' => $review]);

		return $this->setJsonResponse(
			[
				'message'  => $review->state ? Text::_('COM_RADICALREVIEWS_REVIEW_SUCCESS') : Text::_('COM_RADICALREVIEWS_REVIEW_SUCCESS_UNPUBLISHED'),
				'html'     => $html,
				'redirect' => false,
			]
		);
	}

	/**
	 * Method to get checkout data.
	 *
	 * @return array|false Checkout data on success, false on failure.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	protected function getReviewData()
	{
		// Check token
		$this->checkAjaxToken();

		/* @var FormModel $model */
		$model = $this->getModel();

		// Save the data in the session
		$app     = Factory::getApplication();
		$data    = [];
		$allData = $this->input->post->get('jform', [], 'array');
		$files   = $this->input->files->get('jform', array(), 'array');

		// Filter all data as string
		$filter = InputFilter::getInstance();

		foreach ($allData as $key => $value)
		{
			$data[$key] = $filter->clean($value);
		}

		// Set media
		if (!empty($files['media']))
		{
			$data['media'] = array_slice($files['media'], 0, ComponentHelper::getParams('com_radicalreviews')->get('media_count', 3));

			$this->input->post->set('jform', $data);
		}

		// Validate the formData data
		if (!$form = $model->getForm($data, false))
		{
			$this->message = [];
			foreach ($model->getErrors() as $error)
			{
				$this->message[] = ($error instanceof \Exception) ? $error->getMessage() : $error;
			}
			$this->code = 500;

			return false;
		}

		// Set state
		$data['state'] = ComponentHelper::getParams($this->option)->get('state', 0);

		// Test whether the data is valid
		$validData = $model->validate($form, $data);

		if ($validData === false)
		{
			$this->message = [];
			foreach ($model->getErrors() as $error)
			{
				$this->message[] = ($error instanceof \Exception) ? $error->getMessage() : $error;
			}
			$this->code = 500;

			return false;
		}

		// Validate email with dot
		if (isset($data['email']) && !filter_var($data['email'], FILTER_VALIDATE_EMAIL))
		{
			$this->message   = [];
			$this->message[] = Text::_('COM_RADICALREVIEWS_ERROR_ERONG_EMAIL');
			$this->code      = 500;

			return false;
		}

		// Remove url from text
		$data['text'] = preg_replace('/\b(https?|ftp|file):\/\/[-A-Z0-9+&@#\/%?=~_|$!:,.;]*[A-Z0-9+&@#\/%=~_|$]/i', '[' . Text::_('COM_RADICALREVIEWS_ERROR_TEXT_LINK') . ']', $data['text']);

		return $data;
	}

	/**
	 * Proxy for getModel.
	 *
	 * @param   string  $name    The model name.
	 * @param   string  $prefix  The class prefix.
	 * @param   array   $config  The array of possible config values.
	 *
	 * @return  BaseDatabaseModel|FormModel A model object.
	 *
	 * @since  1.0.0
	 */
	public function getModel($name = 'Form', $prefix = 'Site', $config = array('ignore_request' => true))
	{
		return parent::getModel($name, $prefix, $config);
	}
}