<?php
/*
 * @package   pkg_radicalreviews
 * @version   1.1.0
 * @author    Delo Design
 * @copyright Copyright (c) 2023 Delo Design. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://delo-design.ru
 */

namespace Joomla\Component\RadicalReviews\Site\Helper;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Multilanguage;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\LayoutHelper;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;

defined('_JEXEC') or die;

class ReviewsHelper
{

	/**
	 * @since 1.0.0
	 */
	public static $_items;

	/**
	 * Method for render object reviews statistics.
	 *
	 * @param   int  $item_id  The id of the object.
	 * @param   int  $context  The name of context.
	 *
	 * @return  string  Review view link.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public static function renderStats($item_id, $context)
	{
		$stats  = self::getStats($item_id, $context);
		$result = LayoutHelper::render('components.radicalreviews.reviews.stats', $stats);

		return $result;
	}

	/**
	 * Method for render list of reviews.
	 *
	 * @param   int  $item_id  The id of the object.
	 * @param   int  $context  The name of context.
	 * @param   int  $limit    Limit of the list of reviews.
	 *
	 * @return  string  Review view link.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public static function renderReviews($item_id, $context, $limit)
	{
		$reviews     = self::getReviews($item_id, $context, $limit);
		$keyReviewed = 'new_object_review_' . $context . '_' . $item_id;
		$displayData = [
			'item_id'    => $item_id,
			'context'    => $context,
			'reviews'    => $reviews->items,
			'addReview'  => RouteHelper::getFormRoute($item_id, $context),
			'allReviews' => RouteHelper::getReviewsRoute($item_id, $context),
			'total'      => $reviews->total,
			'reviewed'   => Factory::getApplication()->getUserState($keyReviewed) ?? 0
		];

		$result = LayoutHelper::render('components.radicalreviews.reviews.list', $displayData);

		return $result;
	}

	/**
	 * Method to get reviews.
	 *
	 * @param   int  $item_id  The id of the object.
	 * @param   int  $context  The name of context.
	 * @param   int  $limit    Limit of the list of reviews.
	 *
	 * @return  \stdClass  Array of reviews objects.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public static function getReviews($item_id = null, $context = null, $limit = 10)
	{
		$result = new \stdClass();

		// Get model
		$model = Factory::getApplication()->bootComponent('com_radicalreviews')->getMVCFactory()->createModel('Reviews', 'Site', ['ignore_request' => true]);

		$model->setState('params', Factory::getApplication()->getParams());
		$model->setState('object.item_id', $item_id);
		$model->setState('object.context', $context);
		$model->setState('filter.published', 1);
		$model->setState('list.limit', $limit);

		$result->items = $model->getItems();
		$result->total = $model->getTotal();

		return $result;
	}

	/**
	 * Method to get Content reviews statistic.
	 *
	 * @param   int  $item_id  The id of the object.
	 * @param   int  $context  The name of context.
	 *
	 * @return int[] Content reviews stats statistic array.
	 *
	 * @since  1.0.0
	 */
	public static function getStats($item_id = null, $context = null)
	{
		$stats = array(
			'item_id'   => $item_id,
			'link'      => Route::_(RouteHelper::getReviewsRoute($item_id, $context), false),
			'add'       => Route::_(RouteHelper::getFormRoute($item_id, $context), false),
			'addRating' => Route::_(RouteHelper::getAddRating($item_id, $context), false),
			'context'   => $context,
			'reviews'   => 0,
			'stars'     => 0,
			'votes'     => 0,
			'rating'    => 0,
		);

		// Get reviews
		$db               = Factory::getContainer()->get('DatabaseDriver');
		$query            = $db->getQuery(true)
			->select('COUNT(id)')
			->from($db->quoteName('#__radicalreviews_reviews'))
			->where($db->quoteName('item_id') . ' = ' . (int) $item_id)
			->where($db->quoteName('context') . ' = ' . $db->quote($context))
			->where('state = 1');
		$stats['reviews'] = $db->setQuery($query)->loadResult();

		// Get rating
		$query = $db->getQuery(true)
			->select(array('stars', 'votes'))
			->from($db->quoteName('#__radicalreviews_rating'))
			->where($db->quoteName('item_id') . ' = ' . (int) $item_id)
			->where($db->quoteName('context') . ' = ' . $db->quote($context));

		if ($rating = $db->setQuery($query)->loadObject())
		{
			$stats['stars']  = (int) $rating->stars;
			$stats['votes']  = (int) $rating->votes;
			$stats['rating'] = ceil($stats['stars'] / $stats['votes']);
		}

		// Is voted
		$keyVoted       = 'new_object_rating_' . $context . '_' . $item_id;
		$stats['voted'] = Factory::getApplication()->getUserState($keyVoted) ?? 0;

		// Is reviews
		$keyReviewed       = 'new_object_review_' . $context . '_' . $item_id;
		$stats['reviewed'] = Factory::getApplication()->getUserState($keyReviewed) ?? 0;

		return $stats;
	}

	/**
	 * Method for parse object string.
	 *
	 * @param   string  $item  Item string
	 *
	 * @return  \stdClass
	 *
	 * @since  1.0.0
	 */
	public static function parseItem($item)
	{
		if (!isset(self::$_items[$item]))
		{
			$item      = str_replace(':', '-', $item);
			$itemArray = explode('-', $item);
			$result    = new \stdClass();

			$result->item_id   = array_pop($itemArray);
			$result->component = reset($itemArray);
			$result->context   = 'com_' . implode('.', $itemArray);
			$result->view      = array_pop($itemArray);

			self::$_items[$item] = $result;
		}


		return self::$_items[$item];
	}

	/**
	 * Method for clear context.
	 *
	 * @param   string  $context  Context string
	 *
	 * @return  string
	 *
	 * @since  1.0.0
	 */
	public static function clearContext($context)
	{
		return str_replace('com_', '', $context);
	}

	/**
	 * Method for check multilanguage redirect.
	 *
	 * @param   \stdClass  $object  Object
	 *
	 * @return  bool|mixed
	 *
	 * @since  1.1.0
	 */
	public static function checkMultilanguage($object)
	{
		// Check multilang
		if (Multilanguage::isEnabled())
		{
			$app             = Factory::getApplication();
			$language        = $object->language ?? '';
			$currentLanguage = $app->getLanguage()->getTag();
			$type            = $app->input->get('view');
			$url             = '';

			if (!empty($language) && $language !== '*' && $language !== $currentLanguage)
			{
				if ($type === 'review')
				{
					$url = Route::_(RouteHelper::getReviewRoute($app->input->getInt('id'), $object->id, $object->context, $language), false);
				}
				else if ($type === 'reviews')
				{
					$url =  Route::_(RouteHelper::getReviewsRoute($object->id, $object->context, $language), false);
				}

				if ($url)
				{
					$app->redirect($url, 301);
				}
			}
		}

		return true;
	}
}