<?php
/*
 * @package   pkg_radicalreviews
 * @version   1.1.0
 * @author    Delo Design
 * @copyright Copyright (c) 2023 Delo Design. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://delo-design.ru
 */

namespace Joomla\Component\RadicalReviews\Site\Model;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Filesystem\Folder;
use Joomla\CMS\Filesystem\Path;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Model\ItemModel;
use Joomla\CMS\Router\Route;
use Joomla\Component\RadicalReviews\Administrator\Helper\EventHelper;
use Joomla\Component\RadicalReviews\Administrator\Helper\MediaHelper;
use Joomla\Component\RadicalReviews\Site\Helper\ReviewsHelper;
use Joomla\Component\RadicalReviews\Site\Helper\RouteHelper;
use Joomla\Registry\Registry;
use Joomla\Utilities\ArrayHelper;

class ReviewModel extends ItemModel
{
	/**
	 * Model context string.
	 *
	 * @var  string
	 *
	 * @since  1.0.0
	 */
	protected $_context = 'com_radicalreviews.review';

	/**
	 * An comment.
	 *
	 * @var  array
	 *
	 * @since  1.0.0
	 */
	protected $_item = null;

	/**
	 * An object.
	 *
	 * @var  object
	 *
	 * @since  1.0.0
	 */
	protected $_object = null;

	/**
	 * Method to auto-populate the model state.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	protected function populateState()
	{
		$app = Factory::getApplication('site');

		$object = ReviewsHelper::parseItem($app->input->getString('item'));

		// Set request states
		$this->setState('review.id', $app->input->getInt('id'));
		$this->setState('object.item_id', $object->item_id);
		$this->setState('object.context', $object->context);

		// Set params state
		$params = $app->getParams();
		$this->setState('params', $params);

		// Set published && debug state
		if ($app->input->getInt('debug', 0))
		{
			$this->setState('filter.published', array(0, 1));
			$this->setState('debug', 1);
		}
		else
		{
			$this->setState('filter.published', 1);
		}

		parent::populateState();
	}

	/**
	 * Method to get review data.
	 *
	 * @param   integer  $pk  The id of the review.
	 *
	 * @return  object|boolean|\Exception  Review object on success, false or exception on failure.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function getItem($pk = null)
	{
		$pk = (!empty($pk)) ? $pk : (int) $this->getState('review.id');

		if ($this->_item === null) $this->_item = array();
		if (!isset($this->_item[$pk]))
		{
			try
			{
				$db    = $this->getDatabase();
				$query = $db->getQuery(true)
					->select(array('r.*'))
					->from($db->quoteName('#__radicalreviews_reviews', 'r'))
					->where('r.id = ' . (int) $pk);

				// Join over the users
				$query->select(array('u.name as author_name'))
					->leftJoin($db->quoteName('#__users', 'u') . ' ON u.id = r.created_by');

				// Filter by published state
				$published = $this->getState('filter.published');
				if (is_numeric($published))
				{
					$query->where('r.state = ' . (int) $published);
				}
				elseif (is_array($published))
				{
					$published = ArrayHelper::toInteger($published);
					$published = implode(',', $published);

					$query->where('r.state IN (' . $published . ')');
				}

				$data = $db->setQuery($query)->loadObject();
				if (empty($data) || empty($data->id))
				{
					throw new \Exception(Text::_('COM_RADICALREVIEWS_ERROR_REVIEW_NOT_FOUND'), 404);
				}

				// Set link
				$data->link = Route::_(RouteHelper::getReviewRoute($data->id, $data->item_id,
					$data->context), false);

				// Set object
				$data->object = EventHelper::getObject($this, $data->item_id, $data->context);

				// Set author
				if (empty($data->author_name))
				{
					$data->author_name = $data->name;
				}

				// Set source
				$data->source = $data->source ? new Registry($data->source) : false;

				// Set media
				$data->media = array();
				$folder       = 'images/reviews/' . $data->id;
				$path         = Path::clean(JPATH_ROOT . '/' . $folder);
				if (Folder::exists($path))
				{
					$files = Folder::files($path, '^((?!poster).)*$');

					foreach ($files as $file)
					{
						$data->media[] = $folder . '/' . $file;
					}
				}

				if (!empty($data->media))
				{
					if (MediaHelper::getInstance()->checkMedia(JPATH_ROOT . '/' . $data->media[0]) === 'image')
					{
						$data->image = $data->media[0];
					}
//					else
//					{
//						$item->image = basename($item->media[0]) . '_poster.jpg';
//					}
				}

				$this->_item[$pk] = $data;
			}
			catch (\Exception $e)
			{
				if ($e->getCode() == 404)
				{
					throw new \Exception(Text::_($e->getMessage()), 404);
				}
				else
				{
					$this->setError($e);
					$this->_item[$pk] = false;
				}
			}
		}

		return $this->_item[$pk];
	}

	/**
	 * Method to get object data.
	 *
	 * @param   integer  $pk       The id of the object.
	 * @param   string   $context  The context of the object.
	 *
	 * @return  object|boolean|\Exception  Object on success, false or exception on failure.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function getObject($pk = null, $context = null)
	{
		$pk = (!empty($pk)) ? $pk : (int) $this->getState('review.id');

		if ($this->_object === null)
		{
			$this->_object = array();
		}

		if (!isset($this->_object[$pk]))
		{
			$review = $this->getItem($pk);

			$object            = EventHelper::getObject($this, $review->item_id, $review->context, true);
			$object->addReview = Route::_(RouteHelper::getFormRoute($review->item_id, $review->context), false);

			// Set models data
			$this->state->set('object.item_id', $review->item_id);
			$this->state->set('object.context', $review->context);

			$this->_object[$pk] = $object;
		}

		return $this->_object[$pk];
	}
}