<?php

/*
 * @package   pkg_radicalreviews
 * @version   1.1.0
 * @author    Delo Design
 * @copyright Copyright (c) 2023 Delo Design. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://delo-design.ru
 */

namespace Joomla\Component\RadicalReviews\Site\Service;

use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Component\Router\RouterView;
use Joomla\CMS\Component\Router\RouterViewConfiguration;
use Joomla\CMS\Component\Router\Rules\MenuRules;
use Joomla\CMS\Component\Router\Rules\NomenuRules;
use Joomla\CMS\Component\Router\Rules\StandardRules;
use Joomla\CMS\Factory;
use Joomla\CMS\Menu\AbstractMenu;
use Joomla\Component\RadicalReviews\Site\Helper\ReviewsHelper;
use Joomla\Utilities\ArrayHelper;

/**
 * Routing class of com_prayers
 *
 * @since  1.0.0
 */
class Router extends RouterView
{
	/**
	 * Router ids.
	 *
	 * @var  array
	 *
	 * @since  1.0.0
	 */
	protected $_ids = array();

	/**
	 * Router segments.
	 *
	 * @var  array
	 *
	 * @since  1.0.0
	 */
	protected array $_segments = [];

	/**
	 * Router constructor.
	 *
	 * @param   CMSApplication  $app   The application item.
	 * @param   AbstractMenu    $menu  The menu item to work with.
	 *
	 * @since  1.0.0
	 */
	public function __construct($app = null, $menu = null)
	{
		// Root router
		$root = new RouterViewConfiguration('root');
		$this->registerView($root);

		// Reviews router
		$reviews = new RouterViewConfiguration('reviews');
		$reviews->setKey('item')->setParent($root);
		$this->registerView($reviews);

		// Form route
		$form = new RouterViewConfiguration('form');
		$form->setParent($reviews, 'item');
		$this->registerView($form);

		// Review route
		$review = new RouterViewConfiguration('review');
		$review->setKey('id')->setParent($reviews, 'item');
		$this->registerView($review);

		parent::__construct($app, $menu);

		$this->attachRule(new MenuRules($this));
		$this->attachRule(new StandardRules($this));
		$this->attachRule(new NomenuRules($this));
	}

	/**
	 * Method to get the segment(s) for reviews.
	 *
	 * @param   string  $id     ID of the item to retrieve the segments.
	 * @param   array   $query  The request that is built right now.
	 *
	 * @return  array  The segments of this item.
	 *
	 * @since  1.0.0
	 */
	public function getReviewsSegment($id, $query)
	{
		return array((int) $id => $id);
	}


	/**
	 * Method to get the segment(s) for reviews.
	 *
	 * @param   string  $id     ID of the item to retrieve the segments.
	 * @param   array   $query  The request that is built right now.
	 *
	 * @return  array  The segments of this item.
	 *
	 * @since  1.0.0
	 */
	public function getReviewSegment($id, $query)
	{
		return array($id => $id);
	}

	/**
	 * Method to get the id for reviews.
	 *
	 * @param   string  $segment  Segment to retrieve the id.
	 * @param   array   $query    The request that is parsed right now.
	 *
	 * @return  integer|false  The id of this item or false.
	 *
	 * @since  1.0.0
	 */
	public function getReviewsId($segment, $query)
	{
		return $segment;
	}

	/**
	 * Method to get the id for reviews.
	 *
	 * @param   string  $segment  Segment to retrieve the id.
	 * @param   array   $query    The request that is parsed right now.
	 *
	 * @return  integer|false  The id of this item or false.
	 *
	 * @since  1.0.0
	 */
	public function getReviewId($segment, $query)
	{
		if (!empty($segment))
		{
			$hash = md5('review_' . $segment);
			if (!isset($this->_ids[$hash]))
			{
				$db      = Factory::getContainer()->get('DatabaseDriver');
				$dbquery = $db->getQuery(true)
					->select('id')
					->from('#__radicalreviews_reviews')
					->where($db->quoteName('id') . ' = ' . $db->quote($segment));
				$db->setQuery($dbquery);
				$this->_ids[$hash] = (int) $db->loadResult();
			}

			return $this->_ids[$hash];
		}

		return false;
	}

	/**
	 * Build the route for the com_tags component
	 *
	 * @param   array  &$query  An array of URL arguments
	 *
	 * @return  array  The URL arguments to use to assemble the subsequent URL.
	 *
	 * @since   1.0.0
	 */
	public function build(&$query)
	{

		$segments = parent::build($query);

		if ($segments && $segments[0] == 'review')
		{
			unset($query['item']);
		}

		return $segments;
	}

	/**
     * Parse method for URLs
     *
     * @param   array  &$segments  Array of URL string-segments
     *
     * @return  array  Associative array of query values
     *
     * @since   1.0.0
     */
    public function parse(&$segments)
    {
		$varItem = '';

		// Check form view
		if (isset($segments[1]) && $segments[1] === 'form')
		{
			$varItem = $segments[0];
		}

		// Parse route
		$vars = parent::parse($segments);

		// Set item data for form view
		if ($vars['view'] === 'form' && $varItem)
		{
			$vars['item'] = $varItem;
		}

        return $vars;
    }

}
