<?php
/*
 * @package   pkg_radicalreviews
 * @version   1.1.0
 * @author    Delo Design
 * @copyright Copyright (c) 2023 Delo Design. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://delo-design.ru
 */

namespace Joomla\Plugin\System\RadicalReviews\Extension;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Event\AbstractEvent;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Language\Multilanguage;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Menu\AdministratorMenuItem;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Router\Router;
use Joomla\CMS\Router\SiteRouter;
use Joomla\CMS\Session\Session;
use Joomla\CMS\Uri\Uri;
use Joomla\Event\Event;
use Joomla\Event\SubscriberInterface;
use Joomla\Registry\Registry;

class RadicalReviews extends CMSPlugin implements SubscriberInterface
{
	/**
	 * Load the language file on instantiation.
	 *
	 * @var    bool
	 *
	 * @since  1.0.0
	 */
	protected $autoloadLanguage = true;

	/**
	 * Loads the application object.
	 *
	 * @var  \Joomla\CMS\Application\CMSApplication
	 *
	 * @since  1.0.0
	 */
	protected $app = null;

	/**
	 * Loads the database object.
	 *
	 * @var  \Joomla\Database\DatabaseDriver
	 *
	 * @since  1.0.0
	 */
	protected $db = null;

	/**
	 * RadicalReviews component params.
	 *
	 * @var  Registry|null
	 *
	 * @since  1.0.0
	 */
	protected ?Registry $componentParams = null;

	/**
	 * Is RadicalReviews menu items load.
	 *
	 * @var  bool
	 *
	 * @since  1.0.0
	 */
	protected bool $loadAdminMenu = false;

	/**
	 * Is RadicalReviews old menu item remove.
	 *
	 * @var  bool
	 *
	 * @since  1.0.0
	 */
	protected bool $removeAdminMenu = false;

	/**
	 * Returns an array of events this subscriber will listen to.
	 *
	 * @return  array
	 *
	 * @since   1.0.0
	 */
	public static function getSubscribedEvents(): array
	{
		return [
			'onContentPrepareForm'  => 'onContentPrepareForm',
			'onAfterInitialise'     => 'onAfterInitialise',
			'onAfterRender'         => 'onAfterRender',
			'onPreprocessMenuItems' => 'onPreprocessMenuItems',
			'onBeforeRender'        => 'onBeforeRender',
			'onAjaxRadicalreviews'  => 'onAjaxRadicalReviews'
		];
	}

	/**
	 * Plugin constructor
	 *
	 * @access public
	 */
	public function __construct(&$subject, $config = array())
	{
		parent::__construct($subject, $config);

		// Setup language data.
		$this->mode_sef = $this->app->get('sef', 0);
	}

	/**
	 * Listener for the `onBeforeRender` event.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function onBeforeRender()
	{
		if ($this->app->isClient('administrator')
			&& $this->app->input->getCmd('option') === 'com_config'
			&& $this->app->input->getCmd('view') === 'component'
			&& $this->app->input->getCmd('component') === 'com_radicalreviews')
		{
			$assets = $this->app->getDocument()->getWebAssetManager();
			$assets->getRegistry()->addExtensionRegistryFile('com_radicalreviews');
			$assets->useScript('com_radicalreviews.administrator.config');
		}
	}

	/**
	 * Add onjAtomSPrepareForm trigger.
	 *
	 * @param   Form   $form  The form to be altered.
	 * @param   mixed  $data  The associated data for the form.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function onContentPrepareForm(Event $event)
	{
		/** @var Form $form */
		$form     = $event->getArgument(0);
		$formName = $form->getName();
		$data     = $event->getArgument(1);

		if ($formName === 'com_config.component' && $this->app->input->get('component') === 'com_radicalreviews')
		{
			// Process the radical Reviews plugins
			PluginHelper::importPlugin('radicalreviews');

			// Trigger event
			Factory::getApplication()->getDispatcher()->dispatch(
				'onRadicalReviewsConfigForm',
				AbstractEvent::create(
					'onRadicalReviewsConfigForm',
					[
						'subject' => $this,
						'form'    => $form,
						'data'    => $data
					]
				)
			);
		}
	}

	/**
	 * Override classes and set admin cookie.
	 *
	 * @since  1.0.0
	 */
	public function onAfterInitialise()
	{
		if (!$this->app->isClient('site'))
		{
			return;
		}

		// Import RadicalReviews plugin group
		PluginHelper::importPlugin('radicalreviews');
	}

	/**
	 * Method to ajax functions.
	 *
	 * @return mixed Function result.
	 *
	 * @throws  \Exception
	 *
	 * @since  1.0.0
	 */
	public function onAjaxRadicalReviews()
	{
		$action = $this->app->input->get('action');

		if (empty($action) || !method_exists($this, $action))
		{
			// Load reviews component
			$this->app->getLanguage()->load('com_radicalreviews', JPATH_SITE);

			throw new \Exception(Text::_('COM_RADICALREVIEWS_ERROR_METHOD_NOT_FOUND'), 500);
		}

		return $this->$action();
	}

	/**
	 * Method to add rating to object.
	 *
	 * @return bool.
	 *
	 * @throws \Exception
	 *
	 * @since  1.0.0
	 */
	protected function addRating()
	{
		$context = $this->app->input->getString('context');
		$value   = $this->app->input->getInt('value');
		$item_id = $this->app->input->getInt('item_id');
		$rated   = (int) $this->app->getUserState('new_object_rating_' . $context . '_' . $item_id);

		if (empty($value) || empty($item_id) || empty($context) || $rated)
		{
			throw new \Exception(Text::_('COM_RADICALREVIEWS_ERROR_EMPTY_DATA'), 500);
		}

		// Add result
		$db    = $this->db;
		$query = $db->getQuery(true)
			->select('*')
			->from($db->quoteName('#__radicalreviews_rating'))
			->where($db->quoteName('item_id') . ' = ' . $item_id)
			->where($db->quoteName('context') . ' = ' . $db->quote($context));

		if ($update = $db->setQuery($query)->loadObject())
		{
			$update->votes    = (int) $update->votes + 1;
			$update->stars    = (int) $update->stars + $value;
			$update->modified = Factory::getDate()->toSql();

			$db->updateObject('#__radicalreviews_rating', $update, 'id');
		}
		else
		{
			$insert           = new \stdClass();
			$insert->context  = $context;
			$insert->item_id  = $item_id;
			$insert->stars    = $value;
			$insert->votes    = 1;
			$insert->modified = Factory::getDate()->toSql();

			$db->insertObject('#__radicalreviews_rating', $insert);
		}

		// Save value
		$this->app->setUserState('new_object_rating_' . $context . '_' . $item_id, ceil($value));

		return true;
	}

	/**
	 * Method move contentreviews script to footer.
	 *
	 * @since  1.0.0
	 */
	public function onAfterRender()
	{
		if ($this->app->isClient('site'))
		{
			$body = $this->app->getBody();

			if (preg_match('|<head>(.*)</head>|si', $body, $matches))
			{
				$search = $matches[1];
				if (preg_match_all('#<script.*src="(.+?)".*</script>#i', $search, $scripts))
				{
					$replace = $search;
					$footer  = '';
					foreach ($scripts[0] as $s => $script)
					{
						if (preg_match('#com_radicalreviews#', $script))
						{
							$replace = str_replace($script, '', $replace);
							$footer  .= '<script defer src="' . $scripts[1][$s] . '"></script>';
						}
					}

					if (!empty($footer))
					{
						$body = str_replace($search, $replace, $body);
						$body = str_replace('</body>', $footer . '</body>', $body);

						$this->app->setBody($body);
					}
				}
			}
		}
	}

	/**
	 * Method to change Joomla 4 administartor menu.
	 *
	 * @param   string  $context   Context selector string.
	 * @param   array   $children  Menu items array.
	 *
	 * @since  1.0.0
	 */
	public function onPreprocessMenuItems(Event $event)
	{
		$context  = $event->getArgument(0);
		$children = $event->getArgument(1);

		if ($this->app->isClient('administrator') && $context === 'com_menus.administrator.module')
		{
			if ($this->loadAdminMenu === false)
			{
				$parent = new AdministratorMenuItem (array(
					'title'     => 'COM_RADICALREVIEWS_MENUS',
					'type'      => 'component',
					'link'      => 'index.php?option=com_radicalreviews',
					'element'   => 'com_jatoms',
					'class'     => 'class:comments',
					'ajaxbadge' => null,
					'dashboard' => 'radicalreviews'
				));

				// Add reviews view
				$parent->addChild(new AdministratorMenuItem (array(
					'title'     => 'COM_RADICALREVIEWS_MENUS_REVIEWS',
					'type'      => 'component',
					'link'      => 'index.php?option=com_radicalreviews&view=reviews',
					'element'   => 'com_radicalreviews',
					'class'     => '',
					'ajaxbadge' => null,
					'dashboard' => null,
					'scope'     => 'default',
				)));

				// Add rating view
				$parent->addChild(new AdministratorMenuItem (array(
					'title'     => 'COM_RADICALREVIEWS_MENUS_RATING',
					'type'      => 'component',
					'link'      => 'index.php?option=com_radicalreviews&view=rating',
					'element'   => 'com_radicalreviews',
					'class'     => '',
					'ajaxbadge' => null,
					'dashboard' => null,
					'scope'     => 'default',
				)));

				// Add separator
				$parent->addChild(new AdministratorMenuItem (array(
					'title'     => null,
					'type'      => 'separator',
					'element'   => 'com_radicalreviews',
					'class'     => '',
					'ajaxbadge' => null,
					'dashboard' => null
				)));

				// Add config view
				$parent->addChild(new AdministratorMenuItem (array(
					'title'     => 'COM_RADICALREVIEWS_MENUS_CONFIG',
					'type'      => 'component',
					'link'      => 'index.php?option=com_config&view=component&component=com_radicalreviews',
					'element'   => 'com_config',
					'class'     => '',
					'ajaxbadge' => null,
					'dashboard' => null,
					'scope'     => 'default',
				)));

				/* @var $root AdministratorMenuItem */
				$root = $children[0]->getParent();
				$root->addChild($parent);
				$this->loadAdminMenu = true;

			}
			elseif ($this->removeAdminMenu === false)
			{
				foreach ($children as $child)
				{
					if ($child->type === 'component'
						&& (int) $child->component_id === ComponentHelper::getComponent('com_radicalreviews')->id)
					{
						$child->getParent()->removeChild($child);
						$this->removeAdminMenu = true;
					}
				}
			}
		}
	}
}