<?php

namespace MSGJoomla\Plugin\Finder\RadicalMart\Adapter;

use Exception;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\Component\Finder\Administrator\Indexer\Indexer;
use Joomla\Component\RadicalMart\Site\Helper\RouteHelper;
use Joomla\Component\Finder\Administrator\Indexer\Helper;
use Joomla\Component\Finder\Administrator\Indexer\Result;
use Joomla\Database\DatabaseQuery;
use Joomla\Database\QueryInterface;
use Joomla\Registry\Registry;
use MSGJoomla\Plugin\Finder\RadicalMart\Helper\CategoryHelper;

class Product extends Adapter
{
    /**
     * The plugin identifier.
     *
     * @var    string
     * @since  0.0.1
     */
    protected $context = 'RadicalMart';

    /**
     * The component context.
     *
     * @var    string
     * @since  0.0.1
     */
    protected $item_context = 'com_radicalmart.product';

    /**
     * The extension name.
     *
     * @var    string
     * @since  0.0.1
     */
    protected $extension = 'com_radicalmart';

    /**
     * The sublayout to use when rendering the results.
     *
     * @var    string
     * @since  0.0.1
     */
    protected $layout = 'product';

    /**
     * The type of content that the adapter indexes.
     *
     * @var    string
     * @since  0.0.1
     */
    protected $type_title = 'RadicalMart_Product';

    /**
     * The table name.
     *
     * @var    string
     * @since  0.0.1
     */
    protected $table = '#__radicalmart_products';

    /**
     * Product categories array
     *
     * @var  array|null
     *
     * @since  0.0.1
     */
    protected $_categories = null;


    /**
     * Method to index an item. The item must be a Result object.
     *
     * @param   Result  $item  The item to index as a Result object.
     *
     * @return  void
     *
     * @since   0.0.1
     * @throws  Exception on database error.
     */
    protected function index(Result $item)
    {
        $item->setLanguage();

        // Check if the extension is enabled.
        if (ComponentHelper::isEnabled($this->extension) === false) {
            return;
        }

        $globalParams = ComponentHelper::getParams('com_radicalmart');

        $item->context = $this->item_context;

        // Initialise the item parameters.
        $registry     = new Registry($item->params);

        $item->params = clone $globalParams;
        $item->params->merge($registry);

        // Trigger the onContentPrepare event.
        $item->summary = Helper::prepareContent($item->summary, $item->params, $item);
        $item->body    = Helper::prepareContent($item->body, $item->params, $item);

        // Create a URL as identifier to recognise items again.
        $item->url = $this->getUrl($item->id, $this->extension, $this->layout);

        // Build the necessary route and path information.
        $item->route = RouteHelper::getProductRoute($item->id, $item->catid, $item->language);
        $item->access = $this->translateState($item->state);

        // Get the menu title if it exists.
        $title = $this->getItemMenuTitle($item->url);

        // Adjust the title if necessary.
        if (!empty($title) && $this->pluginParams->get('use_menu_title', true)) {
            $item->title = $title;
        }

        // Set media
        $media = new Registry($item->media);
        $gallery = $media->get('gallery', false);

        // Add the image.
        if (!empty($gallery)) {
            foreach ($gallery as $gallery_item) {
                if ($gallery_item->type == 'image') {
                    $item->imageUrl = $gallery_item->src;
                    $item->imageAlt = $gallery_item->alt ?? '';

                    break;
                }
            }
        }

        // Translate the state. Product should only be published if the category is published.
        $item->state = $this->translateState($item->state, $item->cat_state);

        // Add the type taxonomy data.
        $item->addTaxonomy('Type', $this->type_title);
        if (!empty($item->author)) {
            $item->addTaxonomy('Author', $item->author);
        }

        if ($this->pluginParams->get('index_codes', true)) {
            $item->addInstruction(Indexer::TITLE_CONTEXT, 'code');
        }

        if ($this->pluginParams->get('index_category', true)) {
            $item->addInstruction(Indexer::TITLE_CONTEXT, 'cat_title');
        }

        // Add the language taxonomy data.
        $item->addTaxonomy('Language', $item->language);

        // Get content extras.
        Helper::getContentExtras($item);

        // Index the item.
        $this->indexer->index($item);
    }

    /**
     * Method to setup the indexer to be run.
     *
     * @return  boolean  True on success.
     *
     * @throws Exception
     * @since   0.0.1
     */
    protected function setup()
    {
        return true;
    }

    /**
     * Method to get a SQL query to load the published and access states for
     * an article and category.
     *
     * @return  QueryInterface  A database object.
     *
     * @since   0.0.1
     */
    protected function getStateQuery()
    {
        $db = $this->getDatabase();
        $query = $db->getQuery(true);

        // Item ID
        $query->select('a.id');

        // Item and category published state
        $query->select('a.' . $this->state_field . ' AS state, c.state AS cat_state')
            ->from($this->table . ' AS a')
            ->join('LEFT', '#__radicalmart_categories AS c ON c.id = a.category');

        return $query;
    }

    /**
     * Method to get the SQL query used to retrieve the list of content items.
     *
     * @param   mixed  $query  A DatabaseQuery object or null.
     *
     * @return  DatabaseQuery  A database object.
     *
     * @since   0.0.1
     */
    protected function getListQuery($query = null)
    {
        $db = $this->getDatabase();

        // Check if we can use the supplied SQL query.
        $query = $query instanceof DatabaseQuery ? $query : $db->getQuery(true)
            ->select('a.id, a.title, a.alias, a.code, a.introtext AS summary, a.fulltext AS body')
            ->select('a.media')
            ->select('a.state, a.category as catid, a.created AS start_date, a.created_by')
            ->select('a.modified, a.modified_by, a.params')
            ->select('a.language, a.ordering')
            ->select('c.title AS cat_title, c.state AS cat_state');

        // Handle the alias CASE WHEN portion of the query
        $case_when_item_alias = ' CASE WHEN ';
        $case_when_item_alias .= $query->charLength('a.alias', '!=', '0');
        $case_when_item_alias .= ' THEN ';
        $a_id = $query->castAsChar('a.id');
        $case_when_item_alias .= $query->concatenate([$a_id, 'a.alias'], ':');
        $case_when_item_alias .= ' ELSE ';
        $case_when_item_alias .= $a_id . ' END as slug';
        $query->select($case_when_item_alias);

        $case_when_category_alias = ' CASE WHEN ';
        $case_when_category_alias .= $query->charLength('c.alias', '!=', '0');
        $case_when_category_alias .= ' THEN ';
        $c_id = $query->castAsChar('c.id');
        $case_when_category_alias .= $query->concatenate([$c_id, 'c.alias'], ':');
        $case_when_category_alias .= ' ELSE ';
        $case_when_category_alias .= $c_id . ' END as catslug';
        $query->select($case_when_category_alias)

            ->select('u.name AS author')
            ->from('#__radicalmart_products AS a')
            ->join('LEFT', '#__radicalmart_categories AS c ON c.id = a.category')
            ->join('LEFT', '#__users AS u ON u.id = a.created_by');

        return $query;
    }
}