<?php

namespace MSGJoomla\Plugin\Finder\RadicalMart\Extension;

use Exception;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\MVC\Model\ModelInterface;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Table\Table;
use Joomla\Component\Content\Site\Helper\RouteHelper;
use Joomla\Component\Finder\Administrator\Indexer\Helper;
use Joomla\Component\Finder\Administrator\Indexer\Indexer;
use Joomla\Component\Finder\Administrator\Indexer\Result;
use Joomla\Component\RadicalMartImport\Administrator\Driver\DriverFactory;
use Joomla\Database\DatabaseAwareInterface;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Database\DatabaseQuery;
use Joomla\Database\QueryInterface;
use Joomla\Registry\Registry;
use Joomla\Utilities\ArrayHelper;
use MSGJoomla\Plugin\Finder\RadicalMart\Adapter\Adapter;
use MSGJoomla\Plugin\Finder\RadicalMart\Adapter\AdapterFactory;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Smart Search plugin for RadicalMart.
 *
 * @since  0.0.1
 */
final class Radicalmart extends CMSPlugin implements DatabaseAwareInterface
{
    use DatabaseAwareTrait;

    /**
     * Load the language file on instantiation.
     *
     * @var    boolean
     * @since  0.0.1
     */
    protected $autoloadLanguage = true;

    /**
     * Construct
     *
     * @param object $subject The object to observe.
     * @param array $config An array that holds the plugin configuration.
     *
     * @since   0.0.1
     */
    public function __construct(&$subject, $config)
    {
        // Call the parent constructor.
        parent::__construct($subject, $config);
    }

    /**
     * Method to instantiate the indexer adapter.
     *
     * @since   0.0.1
     */
    public function getAdapter($context)
    {
        $parts = explode('.', $context);

        if ($parts[0] !== 'com_radicalmart') {
            return false;
        }

        return AdapterFactory::getAdapter($parts[1], $this->params, $this->getDatabase());
    }

    /**
     * Smart Search after save item method.
     * Reindexes the link information for an item that has been saved.
     *
     * @param string $context The context of the content passed to the plugin.
     * @param Table $row A Table object.
     * @param boolean $isNew True if the content has just been created.
     *
     * @return  void
     *
     * @throws \Exception
     * @since   0.0.1
     */
    public function onFinderAfterSave($context, $row, $isNew): void
    {
        // We only want to handle product here.
        if ($context === 'com_radicalmart.product') {
            /** @var Adapter $adapter */
            if($adapter = $this->getAdapter('com_radicalmart.product')) {
                $adapter->reindex($row->id);
            }
        }

        // We only want to handle category here.
        if ($context === 'com_radicalmart.category') {
            /** @var Adapter $adapter */
            if($adapter = $this->getAdapter('com_radicalmart.product')) {
                $adapter->categoryStateChange(array($row->id), $row->state);
            }
        }
    }

    /**
     * Method to remove the link information for items that have been deleted.
     *
     * @param   string  $context  The context of the action being performed.
     * @param   Table   $table    A Table object containing the record to be deleted
     *
     * @return  void
     *
     * @since   0.0.1
     * @throws  Exception on database error.
     */
    public function onFinderAfterDelete(string $context, Table $table): void
    {
        if ($context === 'com_radicalmart.product') {
            $id = $table->id;
        } elseif ($context === 'com_finder.index') {
            $id = $table->link_id;
        } else {
            return;
        }

        /** @var Adapter $adapter */
        if($adapter = $this->getAdapter($context)) {
            $adapter->remove($id);
        }
    }

    /**
     * Method to update the link information for items that have been changed
     * from outside the edit screen. This is fired when the item is published,
     * unpublished, archived, or unarchived from the list view.
     *
     * @param string $context The context for the content passed to the plugin.
     * @param array $pks An array of primary key ids of the content that has changed state.
     * @param integer $value The value of the state that the content has been changed to.
     *
     * @return  void
     *
     * @throws Exception
     * @since   0.0.1
     */
    public function onFinderChangeState($context, $pks, $value)
    {
        // We only want to handle product here.
        if ($context === 'com_radicalmart.product') {
            /** @var Adapter $adapter */
            if($adapter = $this->getAdapter('com_radicalmart.product')) {
                $adapter->itemStateChange($pks, $value);
            }
        }

        // We only want to handle category here.
        if ($context === 'com_radicalmart.category' && $value === 0) {
            /** @var Adapter $adapter */
            if($adapter = $this->getAdapter('com_radicalmart.product')) {
                $adapter->categoryStateChange($pks, $value);
            }
        }

        // Handle when the plugin is disabled.
        if ($context === 'com_plugins.plugin' && $value === 0) {
            /** @var Adapter $adapter */
            if($adapter = $this->getAdapter('com_radicalmart.product')) {
                $adapter->pluginDisable($pks);
            }
        }
    }

    /**
     * Method to get the adapter state and push it into the indexer.
     *
     * @return  void
     *
     * @since   0.0.1
     * @throws  Exception on error.
     */
    public function onStartIndex(): void
    {
        /** @var Adapter $adapter */
        if($adapter = $this->getAdapter('com_radicalmart.product')) {
            $adapter->startIndex();
        }
    }

    /**
     * Method to prepare for the indexer to be run. This method will often
     * be used to include dependencies and things of that nature.
     *
     * @return  boolean  True on success.
     *
     * @since   0.0.1
     * @throws  Exception on error.
     */
    public function onBeforeIndex()
    {
        /** @var Adapter $adapter */
        if($adapter = $this->getAdapter('com_radicalmart.product')) {
            return $adapter->beforeIndex();
        }

        return false;
    }

    /**
     * Method to index a batch of content items. This method can be called by
     * the indexer many times throughout the indexing process depending on how
     * much content is available for indexing. It is important to track the
     * progress correctly so we can display it to the user.
     *
     * @return  boolean  True on success.
     *
     * @since   0.0.1
     * @throws  Exception on error.
     */
    public function onBuildIndex()
    {
        /** @var Adapter $adapter */
        if($adapter = $this->getAdapter('com_radicalmart.product')) {
            return $adapter->buildIndex();
        }

        return false;
    }

    /**
     * Method to remove outdated index entries
     *
     * @return  integer
     *
     * @since   0.0.1
     */
    public function onFinderGarbageCollection()
    {
        /** @var Adapter $adapter */
        if($adapter = $this->getAdapter('com_radicalmart.product')) {
            return $adapter->finderGarbageCollection();
        }

        return 0;
    }
}
