<?php
/*
 * @package     RadicalMart Package
 * @subpackage  plg_system_radicalmart
 * @version     __DEPLOY_VERSION__
 * @author      RadicalMart Team - radicalmart.ru
 * @copyright   Copyright (c) 2024 RadicalMart. All rights reserved.
 * @license     GNU/GPL license: https://www.gnu.org/copyleft/gpl.html
 * @link        https://radicalmart.ru/
 */

namespace Joomla\Plugin\RadicalMart\Fiscalization\Extension;

\defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Form\FormFactoryInterface;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Component\RadicalMart\Administrator\Helper\PriceHelper as RadicalMartPriceHelper;
use Joomla\Component\RadicalMartExpress\Administrator\Helper\ParamsHelper as RadicalMartExpressParamsHelper;
use Joomla\Event\SubscriberInterface;
use Joomla\Plugin\RadicalMart\Fiscalization\Helper\ParamsHelper;
use Joomla\Plugin\RadicalMart\Fiscalization\Helper\PriceHelper;
use Joomla\Registry\Registry;

class Fiscalization extends CMSPlugin implements SubscriberInterface
{
	/**
	 * Load the language file on instantiation.
	 *
	 * @var    bool
	 *
	 * @since  __DEPLOY_VERSION__
	 */
	protected $autoloadLanguage = true;

	/**
	 * Loads the application object.
	 *
	 * @var  \Joomla\CMS\Application\CMSApplication
	 *
	 * @since  __DEPLOY_VERSION__
	 */
	protected $app = null;

	/**
	 * Plugins forms path.
	 *
	 * @var    string
	 *
	 * @since  __DEPLOY_VERSION__
	 */
	protected string $formsPath = JPATH_PLUGINS . '/radicalmart/fiscalization/forms';

	/**
	 * Returns an array of events this subscriber will listen to.
	 *
	 * @return  array
	 *
	 * @since   __DEPLOY_VERSION__
	 */
	public static function getSubscribedEvents(): array
	{
		return [
			'onRadicalMartPrepareForm'               => 'onRadicalMartPrepareForm',
			'onRadicalMartPrepareConfigCurrencyForm' => 'onRadicalMartPrepareConfigCurrencyForm',
			'onRadicalMartPreparePricesForm'         => 'onRadicalMartPreparePricesForm',
			'onRadicalMartPrepareOrderProductsForm'  => 'onRadicalMartPrepareOrderProductsForm',
			'onRadicalMartGetOrderForm'              => 'onRadicalMartGetOrderForm',
			'onRadicalMartGetOrder'                  => 'onGetOrder',

			'onRadicalMartExpressPrepareConfigForm'        => 'onRadicalMartExpressPrepareConfigForm',
			'onRadicalMartExpressPrepareProductForm'       => 'onRadicalMartExpressPrepareProductForm',
			'onRadicalMartExpressPrepareOrderProductsForm' => 'onRadicalMartExpressPrepareOrderProductsForm',
			'onRadicalMartExpressGetOrderForm'             => 'onRadicalMartExpressGetOrderForm',
			'onRadicalMartExpressGetOrder'                 => 'onGetOrder',
		];
	}

	/**
	 * Add fiscalization fields to RadicalMart payment method form.
	 *
	 * @param   Form   $form  Price form object.
	 * @param   mixed  $data  The data expected for the form.
	 *
	 * @since __DEPLOY_VERSION__
	 */
	public function onRadicalMartPrepareForm(Form $form, $data)
	{
		if ($form->getName() === 'com_radicalmart.paymentmethod')
		{
			$form->loadFile($this->formsPath . '/radicalmart/paymentmethod.xml');
		}
	}

	/**
	 * Add converter to currency config form to RadicalMart.
	 *
	 * @param   Form  $form  The form to be altered.
	 *
	 * @throws  \Exception
	 *
	 * @since  __DEPLOY_VERSION__
	 */
	public function onRadicalMartPrepareConfigCurrencyForm(Form $form)
	{
		$currencies = RadicalMartPriceHelper::getCurrencies();
		if (count($currencies) > 0)
		{
			$form->loadFile($this->formsPath . '/radicalmart/config_currency.xml');
			$this->loadFiscalizationSubform($form, 'fiscalization_product', null, [
				'context'     => 'com_radicalmart.config',
				'params_type' => 'product',
			]);

			$this->loadFiscalizationSubform($form, 'fiscalization_shipping', null, [
				'context'     => 'com_radicalmart.config',
				'params_type' => 'shipping',
			]);
		}
	}

	/**
	 * Add fiscalization fields to RadicalMart prices forms.
	 *
	 * @param   Form   $form  Price form object.
	 * @param   mixed  $data  The data expected for the form.
	 *
	 * @since __DEPLOY_VERSION__
	 */
	public function onRadicalMartPreparePricesForm(Form $form, $data)
	{
		$formName = $form->getName();
		if (!in_array($formName, [
			'com_radicalmart.product.prices',
			'com_radicalmart.category.prices',
			'com_radicalmart.shippingmethod.prices'
		]))
		{
			return;
		}

		$registry = new Registry($data);
		if ($formName === 'com_radicalmart.product.prices')
		{
			$form->loadFile($this->formsPath . '/radicalmart/product_prices.xml');
			$this->loadFiscalizationSubform($form, 'fiscalization_product', 'prices.{currency_group}', [
				'context'     => 'com_radicalmart.product',
				'params_type' => 'product',
				'item_id'     => $registry->get('id'),
				'category_id' => $registry->get('category'),
				'currency'    => '{currency_code}'
			]);
		}
		elseif ($formName === 'com_radicalmart.category.prices')
		{
			$form->loadFile($this->formsPath . '/radicalmart/category_prices.xml');
			$this->loadFiscalizationSubform($form, 'fiscalization_product', 'prices.{currency_group}', [
				'context'     => 'com_radicalmart.categories',
				'params_type' => 'product',
				'item_id'     => $registry->get('id'),
				'currency'    => '{currency_code}'
			]);
		}
		elseif ($formName === 'com_radicalmart.shippingmethod.prices')
		{
			$form->loadFile($this->formsPath . '/radicalmart/shippingmethod_prices.xml');
			$this->loadFiscalizationSubform($form, 'fiscalization_shipping', 'prices.{currency_group}', [
				'context'     => 'com_radicalmart.shippingmethod',
				'params_type' => 'shipping',
				'item_id'     => $registry->get('id'),
				'currency'    => '{currency_code}'
			]);
		}
	}

	/**
	 * Add fiscalization fields to RadicalMart order products form.
	 *
	 * @param   string|null  $context  Context selector string.
	 * @param   Form         $form     Prosecutes form object.
	 * @param   mixed        $data     The data expected for the form.
	 *
	 * @since __DEPLOY_VERSION__
	 */
	public function onRadicalMartPrepareOrderProductsForm(?string $context, Form $form, $data)
	{
		$form->loadFile($this->formsPath . '/radicalmart/order_products.xml');
		$this->loadFiscalizationSubform($form, 'fiscalization_product', 'plugins', [
			'context'     => 'com_radicalmart.order.products',
			'params_type' => 'product',
			'currency'    => '{currency_code}'
		]);
	}

	/**
	 * Add fiscalization fields to RadicalMart order form.
	 *
	 * @param   string             $context   Context selector string.
	 * @param   Form               $form      Order form object.
	 * @param   array              $formData  Form data array.
	 * @param   array|null|false   $products  Shipping method data.
	 * @param   object|null|false  $shipping  Shipping method data.
	 * @param   object|null|false  $payment   Payment method data.
	 * @param   array              $currency  Order currency data.
	 *
	 * @since __DEPLOY_VERSION__
	 */
	public function onRadicalMartGetOrderForm(string $context, Form $form, array $formData, $products, $shipping,
	                                                 $payment, array $currency)
	{
		$formName = $form->getName();
		if ($formName !== 'com_radicalmart.order')
		{
			return;
		}

		$form->loadFile($this->formsPath . '/radicalmart/order.xml');
		$this->loadFiscalizationSubform($form, 'fiscalization_shipping', 'shipping.price', [
			'context'     => 'com_radicalmart.order.shipping',
			'params_type' => 'shipping',
			'currency'    => '{currency_code}'
		]);
	}

	/**
	 * Method to load RadicalMart & RadicalMart Express configuration form.
	 *
	 * @param   Form   $form  The form to be altered.
	 * @param   mixed  $data  The associated data for the form.
	 *
	 * @throws \Exception
	 *
	 * @since __DEPLOY_VERSION__
	 */
	public function onRadicalMartExpressPrepareConfigForm(Form $form, $data = [])
	{
		$component = $this->app->input->getCmd('component');
		if ($component === 'com_radicalmart_express')
		{
			$form->loadFile($this->formsPath . '/radicalmart_express/config.xml');
			$this->loadFiscalizationSubform($form, 'fiscalization_product', 'currency', [
				'context'     => 'com_radicalmart_express.config',
				'params_type' => 'product',
			]);
			$this->loadFiscalizationSubform($form, 'fiscalization_shipping', 'currency', [
				'context'     => 'com_radicalmart_express.config',
				'params_type' => 'shipping',
			]);
		}
	}

	/**
	 * Add fiscalization fields to RadicalMart prices forms.
	 *
	 * @param   Form   $form  Price form object.
	 * @param   mixed  $data  The data expected for the form.
	 *
	 * @since __DEPLOY_VERSION__
	 */
	public function onRadicalMartExpressPrepareProductForm(Form $form, $data)
	{
		$formName = $form->getName();
		if ($formName === 'com_radicalmart_express.product')
		{
			$form->loadFile($this->formsPath . '/radicalmart_express/product.xml');
			$this->loadFiscalizationSubform($form, 'fiscalization_product', 'price', [
				'context'     => 'com_radicalmart_express.product',
				'params_type' => 'product',
			]);
		}
	}

	/**
	 * Add fiscalization fields to RadicalMart Express order products form.
	 *
	 * @param   string|null  $context  Context selector string.
	 * @param   Form         $form     Prosecutes form object.
	 * @param   mixed        $data     The data expected for the form.
	 *
	 * @since __DEPLOY_VERSION__
	 */
	public function onRadicalMartExpressPrepareOrderProductsForm(?string $context, Form $form, $data)
	{
		$form->loadFile($this->formsPath . '/radicalmart/order_products.xml');
		$this->loadFiscalizationSubform($form, 'fiscalization_product', 'plugins', [
			'context'     => 'com_radicalmart_express.order.products',
			'params_type' => 'product',
		]);
	}

	/**
	 * Add fiscalization fields to RadicalMart Express order form.
	 *
	 * @param   string             $context   Context selector string.
	 * @param   Form               $form      Order form object.
	 * @param   array              $formData  Form data array.
	 * @param   array|null|false   $products  Shipping method data.
	 * @param   object|null|false  $shipping  Shipping method data.
	 * @param   object|null|false  $payment   Payment method data.
	 * @param   array              $currency  Order currency data.
	 *
	 * @since __DEPLOY_VERSION__
	 */
	public function onRadicalMartExpressGetOrderForm(string $context, Form $form, array $formData, $products, $shipping,
	                                                        $payment, array $currency)
	{
		if ((int) RadicalMartExpressParamsHelper::getComponentParams()->get('shipping', 0) === 0)
		{
			return;
		}

		$formName = $form->getName();
		if ($formName !== 'com_radicalmart_express.order')
		{
			return;
		}

		$form->loadFile($this->formsPath . '/radicalmart_express/order.xml');
		$this->loadFiscalizationSubform($form, 'fiscalization_shipping', 'shipping.price', [
			'context'     => 'com_radicalmart_express.order.shipping',
			'params_type' => 'shipping',
		]);
	}

	/**
	 * Method to add receipt data to order object.
	 *
	 * @param   string|null   $context  Context selector string.
	 * @param   object|null|  $order    Order data object.
	 *
	 * @throws \Exception
	 *
	 * @since __DEPLOY_VERSION__
	 */
	public function onGetOrder(?string $context = null, ?object $order = null)
	{
		$orderAmount = PriceHelper::toInteger($order->total['final'], $order->currency);

		$code                       = $order->currency['code'];
		$group                      = (!empty($order->currency['group'])) ? $order->currency['group'] : strtolower($code);
		$receipt                    = new \stdClass();
		$receipt->order_id          = $order->id;
		$receipt->order_number      = $order->number;
		$receipt->order_description = $order->title;
		$receipt->timestamp         = time();
		$receipt->currecny          = $code;
		$receipt->amount            = $order->total['final'];
		$receipt->amount_integer    = $orderAmount;
		$receipt->items             = [];

		// Add products
		$productsAmount = 0;
		$lastProductKey = null;
		foreach ($order->products as $product_key => $product)
		{
			$price         = $product->order['base'];
			$price_integer = PriceHelper::toInteger($price, $order->currency);

			$price_discount         = $product->order['final'];
			$price_discount_integer = PriceHelper::toInteger($price_discount, $order->currency);

			$sum         = $product->order['sum_final'];
			$sum_integer = PriceHelper::toInteger($sum, $order->currency);

			$sum_no_discount         = $product->order['sum_base'];
			$sum_no_discount_integer = PriceHelper::toInteger($sum_no_discount, $order->currency);

			$params = $this->getProductFiscalizationParams($context, $product, $group);

			$key                  = 'product_' . $product_key;
			$receipt->items[$key] = [
				'id'                      => $product->id,
				'key'                     => $product_key,
				'type'                    => 'product',
				'name'                    => $product->title,
				'currency'                => $code,
				'price'                   => $price,
				'price_integer'           => $price_integer,
				'price_discount'          => $price_discount,
				'price_discount_integer'  => $price_discount_integer,
				'quantity'                => $product->order['quantity'],
				'measurement_unit'        => $params->get('measurement_unit'),
				'sum'                     => $sum,
				'sum_integer'             => $sum_integer,
				'sum_no_discount'         => $sum_no_discount,
				'sum_no_discount_integer' => $sum_no_discount_integer,
				'payment_method'          => $params->get('payment_method'),
				'payment_object'          => $params->get('payment_object'),
				'vat'                     => ['type' => $params->get('vat_type')]
			];

			$productsAmount += $sum_integer;
			$lastProductKey = $key;
		}

		// Add shipping
		$shippingAmount = 0;
		if (!empty($order->shipping) && !empty($order->shipping->order)
			&& !empty($order->shipping->order->price)
			&& (!empty($order->shipping->order->price['base']) || !empty($order->shipping->order->price['final']))
		)
		{
			$shipping = $order->shipping;

			$price         = (!empty($shipping->order->price['base']))
				? $shipping->order->price['base'] : $shipping->order->price['final'];
			$price_integer = PriceHelper::toInteger($price, $order->currency);

			$sum         = (!empty($shipping->order->price['final']))
				? $shipping->order->price['final'] : $shipping->order->price['base'];
			$sum_integer = PriceHelper::toInteger($sum, $order->currency);

			$params = $this->getShippingFiscalizationParams($context, $shipping, $group);

			$receipt->items['shipping_' . $shipping->id] = [
				'id'                      => $shipping->id,
				'type'                    => 'shipping',
				'name'                    => (!empty($shipping->order->title)) ? $shipping->order->title : $shipping->title,
				'currency'                => $code,
				'price'                   => $price,
				'price_integer'           => $price_integer,
				'price_discount'          => $price,
				'price_discount_integer'  => $price_integer,
				'quantity'                => 1,
				'measurement_unit'        => $params->get('measurement_unit'),
				'sum'                     => $sum,
				'sum_integer'             => $sum_integer,
				'sum_no_discount'         => $sum,
				'sum_no_discount_integer' => $sum_integer,
				'payment_method'          => $params->get('payment_method'),
				'payment_object'          => $params->get('payment_object'),
				'vat'                     => ['type' => $params->get('vat_type')]
			];

			$shippingAmount = $sum_integer;
		}

		// Discount balancer
		$discountBalancer = true;
		if (strpos($context, 'com_radicalmart.') !== false)
		{
			$discountBalancer = (!empty($order->payment)
				&& (int) $order->payment->params->get('fiscalization_order_discount_fix', 1) === 1);

		}
		elseif (strpos($context, 'com_radicalmart_express.') !== false)
		{
			$discountBalancer = ((int) ParamsHelper::getRadicalmartExpressComponentParams()
					->get('fiscalization_order_discount_fix', 1) === 1);
		}

		if (!$discountBalancer)
		{
			$order->receipt = $receipt;

			return;
		}

		$itemsAmount = $productsAmount + $shippingAmount;
		if ($orderAmount < $itemsAmount)
		{
			$needle      = $orderAmount - $shippingAmount;
			$coefficient = ($productsAmount - $needle) / $productsAmount;

			$current = 0;
			foreach ($receipt->items as $i => &$item)
			{
				if ($item['type'] !== 'product')
				{
					continue;
				}

				if ($i !== $lastProductKey)
				{
					$sum_integer = ceil($item['sum_integer'] - ($item['sum_integer'] * $coefficient));
					$current     += $sum_integer;
				}
				else
				{
					$sum_integer = $orderAmount - $shippingAmount - $current;
				}

				$sum = PriceHelper::toFloat($sum_integer, $order->currency);

				$price_discount_integer = round($sum_integer / $item['quantity']);
				$price_discount         = PriceHelper::toFloat($price_discount_integer, $order->currency);
				if ($sum_integer !== ($price_discount_integer * $item['quantity']))
				{
					$item['discount_price_incorrect'] = true;
				}

				$item['sum']                    = $sum;
				$item['sum_integer']            = $sum_integer;
				$item['price_discount']         = $price_discount;
				$item['price_discount_integer'] = $price_discount_integer;
			}
		}

		$order->receipt = $receipt;
	}

	/**
	 * Method to get product fiscalization params
	 *
	 * @param   string|null   $context  Context selector string.
	 * @param   object|null|  $product  Order data object.
	 * @param   string|null   $group    Currency group for RadicaMart.
	 *
	 * @throws \Exception
	 *
	 * @return Registry Product fiscalization params.
	 *
	 * @since __DEPLOY_VERSION__
	 */
	public function getProductFiscalizationParams(?string $context = null, ?object $product = null, ?string $group = null): Registry
	{
		$result = ParamsHelper::getDefault('product');
		if (empty($context) || empty($product))
		{
			return $result;
		}
		$result->set('measurement_unit', $product->params->get('quantity_units', 'pcs'));

		$category  = [];
		$component = [];
		if (strpos($context, 'com_radicalmart.') !== false)
		{
			$component = ParamsHelper::getRadicalmartComponentParams($group)->toArray();

			if (!empty($product->category->prices)
				&& !empty($product->category->prices[$group])
				&& !empty($product->category->prices[$group]['fiscalization_product'])
				&& is_array($product->category->prices[$group]['fiscalization_product']))
			{
				$category = $product->category->prices[$group]['fiscalization_product'];
			}
		}
		elseif (strpos($context, 'com_radicalmart_express.') !== false)
		{
			$component = ParamsHelper::getRadicalmartExpressComponentParams()->toArray();
		}

		if (!empty($component))
		{
			foreach ($component as $path => $value)
			{
				if (!empty($value))
				{
					$result->set($path, $value);
				}
			}
		}

		if (!empty($category))
		{
			foreach ($category as $path => $value)
			{
				if (!empty($value))
				{
					$result->set($path, $value);
				}
			}
		}

		if (!empty($product->price['fiscalization_product']) && is_array($product->price['fiscalization_product']))
		{
			foreach ($product->price['fiscalization_product'] as $path => $value)
			{
				if (!empty($value))
				{
					$result->set($path, $value);
				}
			}
		}

		if (!empty($product->order['plugins']) && !empty($product->order['plugins']['fiscalization_product'])
			&& is_array($product->order['plugins']['fiscalization_product']))
		{
			foreach ($product->order['plugins']['fiscalization_product'] as $path => $value)
			{
				if (!empty($value))
				{
					$result->set($path, $value);
				}
			}
		}

		return $result;
	}

	/**
	 * Method to get product fiscalization params
	 *
	 * @param   string|null   $context   Context selector string.
	 * @param   object|null|  $shipping  Shipping data object.
	 * @param   string|null   $currency  Currency group for RadicaMart.
	 *
	 * @throws \Exception
	 *
	 * @return Registry Shipping fiscalization params.
	 *
	 * @since __DEPLOY_VERSION__
	 */
	public function getShippingFiscalizationParams(?string $context = null, ?object $shipping = null, ?string $currency = null): Registry
	{
		$result = ParamsHelper::getDefault('shipping');
		if (empty($context) || empty($shipping))
		{
			return $result;
		}
		$result->set('measurement_unit', 'pcs');

		$component = [];
		$method    = [];
		if (strpos($context, 'com_radicalmart.') !== false)
		{
			$component = ParamsHelper::getRadicalmartComponentParams($currency, 'shipping')->toArray();
			if (!empty($shipping->prices)
				&& !empty($shipping->prices[$currency])
				&& !empty($shipping->prices[$currency]['fiscalization_shipping'])
				&& is_array($shipping->prices[$currency]['fiscalization_shipping']))
			{
				$method = $shipping->prices[$currency]['fiscalization_shipping'];
			}
		}
		elseif (strpos($context, 'com_radicalmart_express.') !== false)
		{
			$component = ParamsHelper::getRadicalmartExpressComponentParams('shipping')->toArray();
		}

		if (!empty($component))
		{
			foreach ($component as $path => $value)
			{
				if (!empty($value))
				{
					$result->set($path, $value);
				}
			}
		}

		if (!empty($method))
		{
			foreach ($method as $path => $value)
			{
				if (!empty($value))
				{
					$result->set($path, $value);
				}
			}
		}

		if (!empty($shipping->order->price) && !empty($shipping->order->price['fiscalization_shipping'])
			&& is_array($shipping->order->price['fiscalization_shipping']))
		{
			foreach ($shipping->order->price['fiscalization_shipping'] as $path => $value)
			{
				if (!empty($value))
				{
					$result->set($path, $value);
				}
			}
		}

		return $result;
	}

	/**
	 * Method to load fiscalization subform.
	 *
	 * @param   Form         $form        Form object.
	 * @param   string|null  $fieldname   Subform field name.
	 * @param   string|null  $group       Subform field group.
	 * @param   array        $attributes  Field attributes.
	 *
	 * @since __DEPLOY_VERSION__
	 */
	protected function loadFiscalizationSubform(Form  $form, ?string $fieldname = null, ?string $group = null,
	                                            array $attributes = [])
	{
		if (empty($fieldname))
		{
			return;
		}

		/** @var Form $fiscalizationForm */
		$fiscalizationForm = Factory::getContainer()->get(FormFactoryInterface::class)
			->createForm('plg_radicalmart_fiscalization.fiscalization');
		$fiscalizationForm->loadFile($this->formsPath . '/fiscalization.xml');

		foreach (['payment_method', 'payment_object', 'vat_type'] as $name)
		{
			foreach ($attributes as $attribute => $value)
			{
				$fiscalizationForm->setFieldAttribute($name, $attribute, $value);
			}
		}

		$form->setFieldAttribute($fieldname, 'formsource', $fiscalizationForm->getXml()->asXML(), $group);
	}
}