<?php
/*
 * @package     RadicalMart Package
 * @subpackage  pkg_radicalmart_fiscalization
 * @version     1.0.2
 * @author      RadicalMart Team - radicalmart.ru
 * @copyright   Copyright (c) 2024 RadicalMart. All rights reserved.
 * @license     GNU/GPL license: https://www.gnu.org/copyleft/gpl.html
 * @link        https://radicalmart.ru/
 */

namespace Joomla\Plugin\RadicalMart\Fiscalization\Field\Fiscalization;

\defined('_JEXEC') or die;

use Joomla\CMS\Form\Field\ListField as BaseField;
use Joomla\CMS\Language\Text;
use Joomla\Component\RadicalMart\Administrator\Helper\PriceHelper as RadicalMartPriceHelper;
use Joomla\Database\ParameterType;
use Joomla\Plugin\RadicalMart\Fiscalization\Helper\ParamsHelper;
use Joomla\Registry\Registry;

class ListField extends BaseField
{
	/**
	 * The form field type.
	 *
	 * @var  string
	 *
	 * @since  1.0.0
	 */
	protected $type = 'fiscalization_list';

	/**
	 * Field form context.
	 *
	 * @var  string
	 *
	 * @since  1.0.0
	 */
	protected $context = null;

	/**
	 * Item category id.
	 *
	 * @var  int
	 *
	 * @since  1.0.0
	 */
	protected $category_id = 0;

	/**
	 * Item id.
	 *
	 * @var  int
	 *
	 * @since  1.0.0
	 */
	protected $item_id = 0;

	/**
	 * Currency data.
	 *
	 * @var  array
	 *
	 * @since  1.0.0
	 */
	protected $currency = null;

	/**
	 * Params type params.
	 *
	 * @var  string
	 *
	 * @since  1.0.0
	 */
	protected $params_type = null;

	/**
	 *  RadicalMart category fiscalization currency params.
	 *
	 * @var  Registry[]|null
	 *
	 * @since  1.0.0
	 */
	public static ?array $_radicalmart_category = null;

	/**
	 * Method to attach a Form object to the field.
	 *
	 * @param   \SimpleXMLElement  $element  The SimpleXMLElement object representing the `<field>` tag.
	 * @param   mixed              $value    The form field value to validate.
	 * @param   string             $group    The field name group control value.
	 *
	 * @return  bool  True on success.
	 *
	 * @since  1.0.0
	 */
	public function setup(\SimpleXMLElement $element, $value, $group = null): bool
	{
		if ($return = parent::setup($element, $value, $group))
		{
			$this->context = (!empty($this->element['context'])) ? (string) $this->element['context']
				: $this->context;

			$this->item_id = (!empty($this->element['item_id'])) ? (int) $this->element['item_id']
				: $this->item_id;

			$this->category_id = (!empty($this->element['category_id'])) ? (int) $this->element['category_id']
				: $this->category_id;

			$currency = (!empty($this->element['currency'])) ? trim((string) $this->element['currency']) : null;
			if (strpos($this->context, 'com_radicalmart.') !== false)
			{
				$this->currency = (!empty($currency)) ? RadicalMartPriceHelper::getCurrency($currency)
					: RadicalMartPriceHelper::getDefaultCurrency();
			}

			$this->params_type = (!empty($this->element['params_type'])) ? (string) $this->element['params_type']
				: $this->params_type;

			if (in_array($this->context, ['com_radicalmart.config', 'com_radicalmart_express.config'])
				&& empty($this->value) && !empty($this->params_type))
			{
				$default = ParamsHelper::getDefault();
				if (isset($default[$this->params_type]))
				{
					$value         = $default[$this->params_type]->get($this->fieldname);
					$this->value   = $value;
					$this->default = $value;
				}
			}

			unset($this->element['useglobal']);
		}

		return $return;
	}

	/**
	 * Method to get the field options.
	 *
	 * @throws  \Exception
	 *
	 * @return  array  The field option objects.
	 *
	 * @since  1.0.0
	 */
	protected function getOptions(): array
	{
		// Prepare options
		$options = parent::getOptions();
		$hint    = $this->hint;
		if (strpos($this->context, 'com_radicalmart.order.') !== false
			|| strpos($this->context, 'com_radicalmart_express.order.') !== false)
		{
			$hint = Text::sprintf('PLG_RADICALMART_FISCALIZATION_PARAMS_FROM_LEGACY');
		}
		elseif ($params = $this->getParamsValue())
		{
			foreach ($options as $option)
			{
				if ((string) $option->value === (string) $params['value'])
				{
					$hint = Text::sprintf('PLG_RADICALMART_FISCALIZATION_PARAMS_FROM_' . $params['type'], $option->text);

					break;
				}
			}
		}

		if (empty($hint))
		{
			$hint = Text::_('PLG_RADICALMART_FISCALIZATION_PARAMS_FROM');
		}

		$default        = new \stdClass();
		$default->value = '';
		$default->text  = $hint;

		array_unshift($options, $default);

		return $options;
	}

	/**
	 * Method to get field params value.
	 *
	 * @throws \Exception
	 *
	 * @return  array  The field params value.
	 *
	 * @since   1.0.0
	 */
	protected function getParamsValue(): array
	{
		$result = [];
		if (empty($this->context))
		{
			return $result;
		}

		if (strpos($this->context, 'com_radicalmart.') !== false)
		{
			$component = ParamsHelper::getRadicalmartComponentParams($this->currency['group'], $this->params_type)
				->get($this->fieldname);

			$result['type']  = 'component';
			$result['value'] = $component;

			if ($this->context === 'com_radicalmart.product' && !empty($this->category_id))
			{
				$category = $this->getRadicalMartCategoryParams($this->category_id, $this->currency['group'])
					->get($this->fieldname);
				if (!empty($category))
				{
					$result['type']  = 'category';
					$result['value'] = $category;
				}
			}
		}

		if (strpos($this->context, 'com_radicalmart_express.') !== false)
		{
			$component       = ParamsHelper::getRadicalmartExpressComponentParams($this->params_type)->get($this->fieldname);
			$result['type']  = 'component';
			$result['value'] = $component;
		}

		return $result;
	}

	/**
	 * Method to get category fiscalization currency params.
	 *
	 * @param   int|null     $pk        Category id.
	 * @param   string|null  $currency  Currency group to get needle params.
	 *
	 * @return   Registry Category fiscalization currency params.
	 *
	 * @since  1.0.0
	 */
	public function getRadicalMartCategoryParams(?int $pk = 0, ?string $currency = null): Registry
	{
		if (empty($pk))
		{
			return new Registry();
		}

		if (self::$_radicalmart_category === null)
		{
			self::$_radicalmart_category = [];
		}

		if (!isset(self::$_radicalmart_category[$pk]))
		{
			$db     = $this->getDatabase();
			$query  = $db->getQuery(true)
				->select('prices')
				->from($db->quoteName('#__radicalmart_categories'))
				->where($db->quoteName('id') . ' = :id')
				->bind(':id', $pk, ParameterType::INTEGER);
			$prices = $db->setQuery($query, 0, 1)->loadResult();
			if (empty($prices))
			{
				self::$_radicalmart_category[$pk] = [];

				return new Registry();
			}

			$prices = (new Registry($prices))->toArray();
			$result = [];
			foreach ($prices as $p => $price)
			{
				if (!isset($result[$p]))
				{
					$result[$p] = new Registry();
				}

				if (!empty($price['fiscalization_product']) && is_array($price['fiscalization_product']))
				{
					foreach ($price['fiscalization_product'] as $path => $value)
					{
						if (empty($value))
						{
							continue;
						}
						$result[$p]->set($path, $value);
					}
				}
			}

			self::$_radicalmart_category[$pk] = $result;
		}

		$category = self::$_radicalmart_category[$pk];

		return (isset($category[$currency])) ? $category[$currency] : new Registry();
	}
}