<?php
/*
 * @package     RadicalMart Package
 * @subpackage  pkg_radicalmart_fiscalization
 * @version     1.0.2
 * @author      RadicalMart Team - radicalmart.ru
 * @copyright   Copyright (c) 2024 RadicalMart. All rights reserved.
 * @license     GNU/GPL license: https://www.gnu.org/copyleft/gpl.html
 * @link        https://radicalmart.ru/
 */

namespace Joomla\Plugin\RadicalMart\Fiscalization\Helper;

\defined('_JEXEC') or die;

use Joomla\Component\RadicalMart\Administrator\Helper\PriceHelper as RadicalMartPriceHelper;
use Joomla\Component\RadicalMartExpress\Administrator\Helper\ParamsHelper as RadicalMartExpressParamsHelper;
use Joomla\Registry\Registry;

class ParamsHelper
{
	/**
	 * RadicalMart global fiscalization currency params.
	 *
	 * @var   Registry[]|null
	 *
	 * @since  1.0.0
	 */
	public static ?array $_default = null;

	/**
	 * RadicalMart global fiscalization currency params.
	 *
	 * @var   Registry[]|null
	 *
	 * @since  1.0.0
	 */
	public static ?array $_radicalmart_component = null;

	/**
	 * RadicalMart Express global fiscalization currency params.
	 *
	 * @var   Registry[]|null
	 *
	 * @since  1.0.0
	 */
	public static ?array $_radicalmart_express_component = null;

	/**
	 * Method to get Radicalmart global fiscalization currency params.
	 *
	 * @param   string|null  $currency  Currency group.
	 * @param   string       $type      Params type (product, shipping).
	 *
	 * @throws \Exception
	 *
	 * @return   Registry Global fiscalization currency params.
	 *
	 * @since  1.0.0
	 */
	public static function getRadicalmartComponentParams(?string $currency = null, string $type = 'product'): Registry
	{
		if (self::$_radicalmart_component === null)
		{
			self::$_radicalmart_component = [];
		}

		if ($currency === null)
		{
			$currency = RadicalMartPriceHelper::getDefaultCurrency()['group'];
		}

		$key = $currency . '_' . $type;
		if (!isset(self::$_radicalmart_component[$key]))
		{
			$result = self::getDefault($type)->toArray();

			$currencyData = RadicalMartPriceHelper::getCurrency(strtoupper($currency));
			if (!empty($currencyData['fiscalization_' . $type]))
			{
				foreach ($currencyData['fiscalization_' . $type] as $path => $value)
				{
					if (empty($value))
					{
						continue;
					}

					$result[$path] = $value;
				}
			}

			self::$_radicalmart_component[$key] = new Registry($result);
		}

		return self::$_radicalmart_component[$key];
	}

	/**
	 * Method to get Radicalmart Express global fiscalization currency params.
	 *
	 * @param   string  $type  Params type (product, shipping).
	 *
	 * @throws \Exception
	 *
	 * @return   Registry Global fiscalization currency params.
	 *
	 * @since  1.0.0
	 */
	public static function getRadicalmartExpressComponentParams(string $type = 'product'): Registry
	{
		if (self::$_radicalmart_express_component === null)
		{
			self::$_radicalmart_express_component = [];
		}

		if (!isset(self::$_radicalmart_express_component[$type]))
		{
			$result       = self::getDefault($type)->toArray();
			$currencyData = (new Registry(RadicalMartExpressParamsHelper::getComponentParams()
				->get('currency', '')))->toArray();
			if (!empty($currencyData['fiscalization_' . $type]))
			{
				foreach ($currencyData['fiscalization_' . $type] as $path => $value)
				{
					if (empty($value))
					{
						continue;
					}

					$result[$path] = $value;
				}
			}

			self::$_radicalmart_express_component[$type] = new Registry($result);
		}

		return self::$_radicalmart_express_component[$type];
	}

	/**
	 * Method to get default fiscalization params.
	 *
	 * @param   string|null  $type  Params type (product, shipping).
	 *
	 * @return   Registry|Registry[] All default fiscalization params if empty type, default params for type if not.
	 *
	 * @since  1.0.0
	 */
	public static function getDefault(?string $type = null)
	{
		if (self::$_default === null)
		{
			self::$_default = [
				'product'  => new Registry([
					'payment_method'   => 'full_payment',
					'payment_object'   => 'commodity',
					'vat_type'         => 'none',
					'measurement_unit' => 'pcs',
				]),
				'shipping' => new Registry([
					'payment_method'   => 'full_payment',
					'payment_object'   => 'service',
					'vat_type'         => 'none',
					'measurement_unit' => 'pcs'
				]),
			];
		}

		if (empty($type))
		{
			return self::$_default;
		}

		return (isset(self::$_default[$type])) ? self::$_default[$type] : self::$_default['product'];

	}
}