<?php
/*
 * @package   plg_system_accesspro
 * @version   1.0.0
 * @author    Dmitriy Vasyukov - https://fictionlabs.ru
 * @copyright Copyright (c) 2023 Fictionlabs. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://fictionlabs.ru/
 */

namespace Joomla\Plugin\System\AccessPro\Rule\Collection;

use Joomla\CMS\Factory;
use Joomla\Plugin\System\AccessPro\Rule\AbstractRule;

class UserLevelRule extends AbstractRule
{
	/**
	 * Type of rule
	 *
	 * @var string
	 *
	 * @since 1.0.0
	 */
	public $type = 'user_level';

	/**
	 * User access levels
	 *
	 * @var string
	 *
	 * @since 1.0.0
	 */
	static public $accessLevels = [];

	/**
	 * Method for check access for element
	 *
	 * @var array $props Properties of elements
	 *
	 * @since 1.0.0
	 */
	public function canAccess($props): bool
	{
		$user       = Factory::getApplication()->getIdentity();
		$levels     = $props[$this->type] ?? [];
		$userLevels = $user->getAuthorisedViewLevels();
		$diffLevels = array_diff($levels, $userLevels);

		return count($diffLevels) < count($levels);
	}

	/**
	 * Method for get fields for dynamic adding to config
	 *
	 * @return array
	 *
	 * @since 1.0.0
	 */
	public function getFields(): array
	{
		return [
			'label'       => $this->type . '_label',
			'type'        => 'select',
			'source'      => true,
			'description' => $this->type . '_description',
			'attrs'       => [
				'multiple' => true,
				'class'    => 'uk-height-small uk-resize-vertical'
			],
			'options'     => $this->getUserAccessLevels()
		];
	}

	/**
	 * Method for user access levels
	 *
	 * @return array
	 *
	 * @since 1.0.0
	 */
	protected function getUserAccessLevels(): array
	{
		$db    = Factory::getContainer()->get('DatabaseDriver');
		$query = $db->getQuery(true);

		if (empty(self::$accessLevels))
		{
			$query->select($db->quoteName('a.id', 'value'));
			$query->select($db->quoteName('a.title', 'text'));
			$query->from($db->quoteName('#__viewlevels', 'a'));
			$query->group($db->quoteName(array('a.id', 'a.title', 'a.ordering')));
			$query->order('a.ordering ASC');

			$db->setQuery($query);

			// Get the user access levels from the database.
			foreach ($db->loadObjectList() as $access)
			{
				self::$accessLevels[$access->text] = $access->value;
			}
		}

		return self::$accessLevels;
	}
}
