<?php
/*
 * @package   plg_system_accesspro
 * @version   1.0.0
 * @author    Dmitriy Vasyukov - https://fictionlabs.ru
 * @copyright Copyright (c) 2023 Fictionlabs. All rights reserved.
 * @license   GNU/GPL license: http://www.gnu.org/copyleft/gpl.html
 * @link      https://fictionlabs.ru/
 */

namespace Joomla\Plugin\System\AccessPro\Rule\Collection;

use Joomla\CMS\Factory;
use Joomla\Plugin\System\AccessPro\Rule\AbstractRule;

class UserGroupRule extends AbstractRule
{
	/**
	 * Type of rule
	 *
	 * @var string
	 *
	 * @since 1.0.0
	 */
	public $type = 'user_group';

	/**
	 * User group levels
	 *
	 * @var string
	 *
	 * @since 1.0.0
	 */
	static public $groupLevels = [];

	/**
	 * Method for check access for element
	 *
	 * @var array $props Properties of elements
	 *
	 * @since 1.0.0
	 */
	public function canAccess($props): bool
	{
		$user   = Factory::getApplication()->getIdentity();
		$levels = $props[$this->type];

		if ($levels)
		{
			foreach ($levels as $level)
			{
				if (in_array($level, $user->groups))
				{
					return true;
				}
			}

			return false;
		}

		return true;
	}

	/**
	 * Method for get fields for dynamic adding to config
	 *
	 * @since 1.0.0
	 */
	public function getFields(): array
	{
		return [
			'label'       => $this->type . '_label',
			'type'        => 'select',
			'source'      => true,
			'description' => $this->type . '_description',
			'attrs'       => [
				'multiple' => true,
				'class'    => 'uk-height-small uk-resize-vertical'
			],
			'options'     => $this->getUserGroupLevels()
		];
	}

	/**
	 * Method for user group levels
	 *
	 * @return array
	 *
	 * @since 1.0.0
	 */
	protected function getUserGroupLevels(): array
	{
		$db    = Factory::getContainer()->get('DatabaseDriver');
		$query = $db->getQuery(true);

		if (empty(self::$groupLevels))
		{
			$query->select($db->quoteName(array('a.id', 'a.title')));
			$query->select($db->quoteName('a.parent_id', 'parent'));
			$query->select('COUNT(DISTINCT b.id) AS level');
			$query->join('LEFT', $db->quoteName('#__usergroups', 'b') . ' ON ' . $db->quoteName('a.lft') . ' > ' . $db->quoteName('b.lft') . ' AND ' . $db->quoteName('a.rgt') . ' > ' . $db->quoteName('b.rgt'));
			$query->from($db->quoteName('#__usergroups', 'a'));
			$query->group($db->quoteName('a.id'));
			$query->order('a.lft ASC');

			$db->setQuery($query);

			// Get the user group levels from the database.
			foreach ($db->loadObjectList() as $group)
			{
				$item_name = $group->title;

				for ($i = 1; $i <= $group->level; $i++)
				{
					$item_name = '| - ' . $item_name;
				}

				self::$groupLevels[$item_name] = $group->id;
			}
		}

		return self::$groupLevels;
	}
}
