<?php
/*
 * @package    TelePost System Plugin
 * @version     1.0.0
 * @author      CaveDesign Studio - cavedesign.ru
 * @copyright   Copyright (c) 2025 CaveDesign Studio. All Rights Reserved.
 * @license     GNU/GPL license: https://www.gnu.org/copyleft/gpl.html
 * @link        https://cavedesign.ru/
 */

namespace Joomla\Plugin\System\Telepost\Extension;

\defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Filter\OutputFilter;
use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\User\UserFactoryInterface;
use Joomla\Component\Content\Administrator\Model\ArticleModel;
use Joomla\Database\DatabaseInterface;
use Joomla\Database\ParameterType;
use Joomla\Event\Event;
use Joomla\Plugin\System\Telepost\Helper\TelepostHelper;
class Telepost extends CMSPlugin
{
	protected $fieldsToSave = [];

	/**
	 * Constructor.
	 *
	 * @param   object  &$subject  The object to observe
	 * @param   array    $config   An optional associative array of configuration settings.
	 *
	 * @since   1.0.0
	 */
	public function __construct(&$subject, array $config)
	{
		parent::__construct($subject, $config);

		$logLevel = $this->params->get('debug_mode', 0)
			? Log::ALL
			: Log::ERROR;
		Log::addLogger(
			['text_file' => 'plg_system_telepost.log.php'],
			$logLevel,
			['telepost']
		);
	}

	/**
	 * AJAX entry point for handling webhook management and incoming data.
	 *
	 * This method acts as a router for all AJAX tasks related to the Telepost plugin.
	 * It performs security checks to ensure the request is intended for this plugin
	 * and contains a valid secret key. Based on the 'task' parameter, it dispatches
	 * the request to the appropriate handler.
	 *
	 * @throws \Exception
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function onAjaxTelepost(): void
	{
		Factory::getApplication()->getLanguage()->load('plg_system_telepost',
			JPATH_ADMINISTRATOR);

		$app    = $this->getApplication();
		$input  = $app->getInput();
		$plugin = $input->getCmd('plugin');
		$task   = $input->getCmd('task');

		if ($plugin !== 'telepost') return;

		$key = $input->getString('key');
		if ($key !== $this->params->get('secret_key'))
		{
			if ($this->params->get('debug_mode'))
			{
				Log::add('Access denied. Invalid secret key provided: ' . $key,
					Log::WARNING, 'telepost');
			}
			$app->setHeader('HTTP/1.1', '403 Forbidden', true)->sendHeaders()->close();
		}

		switch ($task)
		{
			case 'webhook':
				$this->handleWebhook();
				break;
			case 'setwebhook':
				$this->setWebhook();
				break;
			case 'delwebhook':
				$this->deleteWebhook();
				break;
		}
	}

	/**
	 * Handles the incoming webhook request from the Telegram API.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	private function handleWebhook(): void
	{
		$rawInput = file_get_contents('php://input');
		$data     = json_decode($rawInput, true);

		Log::add('Webhook handle started', Log::DEBUG, 'telepost');

		if ($this->params->get('debug_mode'))
		{
			Log::add('INCOMING PAYLOAD: ' . json_encode($data,
					JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT), Log::DEBUG, 'telepost');
		}

		if (!empty($data['channel_post']))
		{
			$this->processChannelPost($data['channel_post']);
		}
		elseif ($this->params->get('debug_mode'))
		{
			Log::add('Event ignored (not a channel_post). Keys found: ' . implode(', ',
					array_keys($data)), Log::DEBUG, 'telepost');
		}

		$this->getApplication()->close(200);
	}

	/**
	 * Sets the Telegram webhook.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	private function setWebhook(): void
	{
		$this->sendWebhookCommand('set');
	}

	/**
	 * Deletes the Telegram webhook.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	private function deleteWebhook(): void
	{
		$this->sendWebhookCommand('delete');
	}

	/**
	 * Sends a command to the Telegram API to set or delete the webhook.
	 *
	 * @param   string  $cmd  The command to send ('set' or 'delete').
	 *
	 * @throws \Exception
	 *
	 * @return  void  Outputs a JSON response and terminates the application.
	 *
	 * @since   1.0.0
	 */
	private function sendWebhookCommand(string $cmd): void
	{
		$app = Factory::getApplication();
		$app->setHeader('Content-Type', 'application/json');
		try
		{
			$token     = $this->params->get('bot_token');
			$secretKey = $this->params->get('secret_key');
			if (!$token || !$secretKey) throw new \RuntimeException('Token or Secret Key is not configured.');

			$webhookUrl = Uri::getInstance()->toString(['scheme', 'host', 'port'])
				. Route::link('site', 'index.php?option=com_ajax&plugin=telepost&group=system&task=webhook&format=raw&key='
					. $secretKey);

			$body = ($cmd === 'set') ? ['url' => $webhookUrl] : [];
			$http = HttpFactory::getHttp();
			$resp = $http->post("https://api.telegram.org/bot{$token}/{$cmd}Webhook", json_encode($body),
				['Content-Type' => 'application/json']);

			if ($resp->code !== 200) throw new \RuntimeException('Telegram API request failed. Code: '
				. $resp->code . '. Body: ' . $resp->body);

			$result = json_decode($resp->body, true);
			if ($result === null) throw new \RuntimeException('Failed to decode JSON from Telegram API.');

			$msg      = $result['ok'] ? "PLG_SYSTEM_TELEPOST_WEBHOOK_" . strtoupper($cmd) . "_OK"
				: ($result['description'] ?? "PLG_SYSTEM_TELEPOST_WEBHOOK_" . strtoupper($cmd) . "_FAIL");
			$response = ['message' => Text::_($msg), 'type' => $result['ok'] ? 'success' : 'error'];
		}
		catch (\Throwable $e)
		{
			$response = ['message' => 'Caught Error: ' . $e->getMessage(), 'type' => 'error'];
		}
		echo json_encode($response);
		$app->close();
	}

	/**
	 * Routes an incoming channel post for processing.
	 *
	 * This method determines if a post is a single message or part of a media group.
	 * It processes single posts directly. For media groups, it only processes the
	 * specific message that contains the 'caption', treating it as the main post
	 * for the entire group. Other parts of the media group are ignored.
	 *
	 * @param   array  $post  The 'channel_post' object from the Telegram API.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	private function processChannelPost(array $post): void
	{
		$mediaGroupId = $post['media_group_id'] ?? null;

		if (!$mediaGroupId)
		{
			$this->createSinglePost($post);

			return;
		}

		if (!empty($post['caption']))
		{
			Log::add("Processing media group {$mediaGroupId} using the post with caption.",
				Log::INFO, 'telepost');
			$this->createSinglePost($post);
		}
		else
		{
			// Debug log for skipped parts of a media group
			if ($this->params->get('debug_mode'))
			{
				Log::add("Skipping media group item {$mediaGroupId} (no caption in this part). Msg ID: " .
					($post['message_id'] ?? '?'), Log::DEBUG, 'telepost');
			}
		}
	}

	/**
	 * Creates a single Joomla article from a Telegram post.
	 *
	 * This is the core method for content creation. It finds a matching rule based on
	 * the channel ID and a required tag in the caption. It then prepares the data,
	 * downloads the main preview image (from a photo or video thumbnail), triggers
	 * the 'onTelepostArticleBeforeSave' event for advanced processing by other plugins,
	 * and finally saves the main article and its custom fields.
	 *
	 * @param   array  $post  The 'channel_post' object from the Telegram API,
	 *                        potentially assembled from a media group.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function createSinglePost(array $post): void
	{
		if ($this->params->get('debug_mode'))
		{
			Log::add('Post Content being processed: ' . json_encode($post, JSON_UNESCAPED_UNICODE),
				Log::DEBUG, 'telepost');
		}

		if (empty($post['caption']))
		{
			Log::add('Post has no caption. Skipping.', Log::INFO, 'telepost');

			return;
		}

		$rules     = (array) $this->params->get('rules');
		$channelId = $post['chat']['id'];
		$caption   = $post['caption'];

		Log::add("Checking rules for Channel ID: {$channelId}. Caption starts with: " . mb_substr($caption, 0, 20) .
			'...', Log::DEBUG, 'telepost');

		foreach ($rules as $rule)
		{
			if ((string) trim($rule->channel_id) !== (string) $channelId) continue;

			$requiredTag = ltrim(trim($rule->filter_tag), '#');
			$searchTag   = '#' . $requiredTag;
			$found       = str_contains($caption, $searchTag);

			if ($this->params->get('debug_mode'))
			{
				Log::add("Comparing with rule tag '{$requiredTag}'. Looking for '{$searchTag}'. 
				Found: " . ($found ? 'YES' : 'NO'), Log::DEBUG, 'telepost');
			}

			if (empty($requiredTag) || !$found) continue;

			Log::add('Matching rule found for tag: ' . $requiredTag, Log::INFO, 'telepost');
			if (empty($rule->category_id) || empty($rule->created_by))
			{
				Log::add('Rule for tag "' . $requiredTag . '" is incomplete. Skipping.', Log::WARNING,
					'telepost');
				continue;
			}

			try
			{
				$user = Factory::getContainer()->get(UserFactoryInterface::class)->loadUserById($rule->created_by);
				if (!$user || !$user->id)
				{
					throw new \Exception('Failed to load user with ID: ' . $rule->created_by);
				}
				Factory::getApplication()->loadIdentity($user);
				Log::add('Identity loaded for user ID: ' . $user->id, Log::DEBUG, 'telepost');
			}
			catch (\Throwable $e)
			{
				Log::add('Critical error setting user identity: ' . $e->getMessage(), Log::CRITICAL,
					'telepost');
				continue;
			}

			$allTags = [];
			if (preg_match_all('/#([\w\p{L}]+)/u', $caption, $matches))
			{
				$allTags = $matches[1];
			}

			$tagIds = [];

			if ($this->params->get('import_tags', 1) && !empty($allTags))
			{

				$aliasesToCheck = [];
				foreach ($allTags as $tagName)
				{
					if (mb_strtolower($tagName) === mb_strtolower($requiredTag)) continue;

					$safeAlias = OutputFilter::stringURLSafe($tagName);
					if (!empty($safeAlias))
					{
						$aliasesToCheck[] = $safeAlias;
					}
				}

				if (!empty($aliasesToCheck))
				{
					$db = Factory::getContainer()->get(DatabaseInterface::class);

					$query = $db->getQuery(true)
						->select($db->quoteName('id'))
						->from($db->quoteName('#__tags'))
						->whereIn($db->quoteName('alias'), $aliasesToCheck, ParameterType::STRING)
						->where($db->quoteName('published') . ' = 1');

					try
					{
						$db->setQuery($query);
						$found = $db->loadColumn();
						if (!empty($found))
						{
							$tagIds = array_map('intval', $found);
							if ($this->params->get('debug_mode'))
							{
								Log::add('Linked existing tags: ' . implode(',', $tagIds),
									Log::DEBUG, 'telepost');
							}
						}
					}
					catch (\Throwable $e)
					{
						Log::add('Tags lookup error: ' . $e->getMessage(), Log::ERROR, 'telepost');
					}
				}
			}

			$cleanCaption = $caption;
			foreach ($allTags as $tag)
			{
				$cleanCaption = preg_replace('/\s*#' . preg_quote($tag, '/') . '/', '', $cleanCaption);
			}
			$cleanCaption = trim($cleanCaption);

			$content      = $this->parseCaption($cleanCaption);
			$title        = $content['title'];
			$text         = $content['text'];

			if (empty($title))
			{
				Log::add('Could not determine title from caption. Skipping this rule.',
					Log::WARNING, 'telepost');
				continue;
			}

			$validDays   = (int) ($rule->valid_days ?? 0);
			$publishDown = Factory::getContainer()->get(DatabaseInterface::class)->getNullDate();
			if ($validDays > 0)
			{
				$date = Factory::getDate();
				$date->modify('+' . $validDays . ' days');
				$publishDown = $date->toSql();
			}

			$backlink = '';
			if ($messageId = $post['message_id'] ?? null)
			{
				if ($chatUsername = $post['chat']['username'] ?? null)
				{
					$backlink = 'https://t.me/' . $chatUsername . '/' . $messageId;
				}
				else
				{
					$chatId          = $post['chat']['id'] ?? null;
					$formattedChatId = substr((string) $chatId, 4);
					$backlink        = 'https://t.me/c/' . $formattedChatId . '/' . $messageId;
				}
			}

			$imagesData      = null;
			$mainImageFileId = null;
			$photos          = $post['photo'] ?? [];

			if (!empty($photos))
			{
				$lastPhoto       = end($photos);
				$mainImageFileId = $lastPhoto['file_id'] ?? null;
			}
			elseif (!empty($post['video']['thumbnail']))
			{
				$mainImageFileId = $post['video']['thumbnail']['file_id'] ?? null;
			}

			if ($mainImageFileId)
			{
				$botToken = $this->params->get('bot_token');
				$imgPath  = $this->params->get('img_path', 'telegram');
				// Security note: TelepostHelper must be updated with allowed extension check
				$filePath = TelepostHelper::downloadTelegramFile($mainImageFileId, $botToken, $imgPath);
				if ($filePath)
				{
					$imagesData = [
						'image_intro'        => $filePath,
						'image_fulltext'     => $filePath,
						'image_intro_alt'    => $title,
						'image_fulltext_alt' => $title,
					];
				}
			}

			$data = [
				'id'           => 0,
				'title'        => $title,
				'alias'        => '',
				'introtext'    => $text,
				'catid'        => (int) $rule->category_id,
				'created_by'   => (int) $rule->created_by,
				'state'        => 1,
				'language'     => '*',
				'publish_down' => $publishDown,
				'tags'         => $tagIds,
				'urls'         => [
					'urla'     => $backlink,
					'urlatext' => 'Источник в Telegram',
					'targeta'  => '_blank',
					'urlb'     => '',
					'urlbtext' => '',
					'targetb'  => '',
					'urlc'     => '',
					'urlctext' => '',
					'targetc'  => ''
				],
			];

			if ($imagesData) $data['images'] = $imagesData;

			PluginHelper::importPlugin('telepost');
			$dataObject = (object) $data;
			$event      = new Event('onTelepostArticleBeforeSave', ['data' => $dataObject, 'post' => $post]);
			$this->getApplication()->getDispatcher()->dispatch($event->getName(), $event);
			$data   = (array) $dataObject;
			$userId = (int) $rule->created_by;

			$customFieldsData = $data['com_fields'] ?? [];

			$mvcFactory = $this->getApplication()->bootComponent('com_content')->getMVCFactory();
			/** @var ArticleModel $model */
			$model = $mvcFactory->createModel('Article', 'Administrator', ['ignore_request' => true]);

			try
			{
				Log::add('Data to save Article: ' . json_encode($data, JSON_UNESCAPED_UNICODE),
					Log::INFO, 'telepost');
				if ($model->save($data))
				{
					$articleId = (int) $model->getState($model->getName() . '.id');
					Log::add('Article successfully created with ID: ' . $articleId,
						Log::INFO, 'telepost');

					if (!empty($customFieldsData))
					{
						Log::add('Send to create Custom Fields: ' . json_encode($customFieldsData,
								JSON_UNESCAPED_UNICODE)
							. ' in ArticleId_' . $articleId, Log::DEBUG, 'telepost');
						$this->saveCustomFields($articleId, $customFieldsData, (int) $rule->created_by);
					}

					return;
				}
				else
				{
					Log::add('model->save() returned false. Errors: ' . print_r($model->getErrors(), true),
						Log::ERROR, 'telepost');
				}
			}
			catch (\Throwable $e)
			{
				Log::add(sprintf('An exception occurred while saving article: %s', $e->getMessage()),
					Log::CRITICAL, 'telepost');
			}
		}
		Log::add('No matching rule found for the combination of channel ID and tags in the caption.',
			Log::INFO, 'telepost');
	}

	/**
	 * Performs basic parsing of a caption string.
	 *
	 * This method extracts a title and main text from a raw caption.
	 * It also applies basic HTML formatting if enabled in params.
	 *
	 * @param   string  $rawCaption  The raw caption text from the Telegram post.
	 *
	 * @return  array   An associative array containing 'title' and 'text'.
	 *
	 * @since   1.0.0
	 */
	protected function parseCaption(string $rawCaption): array
	{
		$parts = explode("\n", $rawCaption, 2);
		$title = trim($parts[0]);

		$text = trim($parts[1] ?? '');

		if (!empty($text))
		{
			$textPartsAfterSplit = explode('***', $text, 2);
			$text                = trim($textPartsAfterSplit[0]);
		}

		if (!empty($text) && $this->params->get('auto_formatting', 1))
		{
			$text = $this->applySimpleFormatting($text);
		}

		return ['title' => $title, 'text' => $text];
	}

	/**
	 * Applies basic paragraph and link formatting.
	 *
	 * @param   string  $text  Plain text.
	 *
	 * @return  string  HTML formatted text.
	 *
	 * @since 1.0.0
	 */
	private function applySimpleFormatting(string $text): string
	{
		$paragraphs = preg_split('/\n\s*\n/', $text);
		$htmlParts  = [];

		foreach ($paragraphs as $paragraph)
		{
			$paragraph = trim($paragraph);
			if (empty($paragraph)) continue;

			$paragraph = $this->autoLink($paragraph);

			$formattedParagraph = nl2br($paragraph, false);

			$htmlParts[] = '<p>' . $formattedParagraph . '</p>';
		}

		return implode('', $htmlParts);
	}

	/**
	 * Converts URLs in text to safe HTML links.
	 *
	 * @param   string  $text
	 *
	 * @return  string
	 *
	 * @since 1.0.0
	 */
	private function autoLink(string $text): string
	{
		$pattern = '~(https?://[^\s]+)~i';

		return preg_replace_callback($pattern, function ($matches) {
			$fullMatch = $matches[0];

			$url                 = rtrim($fullMatch, ".,!?:;\"'");
			$trailingPunctuation = substr($fullMatch, strlen($url));

			$attr = 'target="_blank" rel="nofollow noopener noreferrer"';

			return '<a href="' . $url . '" ' . $attr . '>' . $url . '</a>' . $trailingPunctuation;
		}, $text);
	}

	/**
	 * Saves custom field values by directly inserting into the #__fields_values table.
	 * Handles serialization for complex field types like 'gallery'.
	 *
	 * @param   int    $articleId   The ID of the article.
	 * @param   array  $fieldsData  An associative array of ['field_id' => 'value'].
	 * @param   int    $userId      The ID of the user context.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function saveCustomFields(int $articleId, array $fieldsData, int $userId): void
	{
		if (empty($fieldsData) || $articleId <= 0)
		{
			Log::add('saveCustomFields: No fields data or invalid article ID.',
				Log::DEBUG, 'telepost');

			return;
		}

		try
		{
			$db = Factory::getContainer()->get(DatabaseInterface::class);

			foreach ($fieldsData as $fieldId => $value)
			{
				// 1. Get field type
				$fieldType = null;
				$query     = $db->getQuery(true)
					->select($db->quoteName('type'))
					->from($db->quoteName('#__fields'))
					->where($db->quoteName('id') . ' = :fieldid')
					->bind(':fieldid', $fieldId, ParameterType::INTEGER);
				$db->setQuery($query);
				$fieldType = $db->loadResult();

				if ($fieldType === null)
				{
					Log::add("saveCustomFields: Could not determine type for field ID {$fieldId}. 
					Skipping.", Log::WARNING, 'telepost');
					continue;
				}

				// 2. Process value
				$valueToStore                = $value;
				$typesRequiringSerialization = ['gallery', 'subform', 'repeatable'];

				if (in_array($fieldType, $typesRequiringSerialization))
				{
					if (is_array($value) || is_object($value))
					{
						$valueToStore = json_encode($value,
							JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
						if (json_last_error() !== JSON_ERROR_NONE)
						{
							Log::add("saveCustomFields: Failed to JSON encode value for field ID {$fieldId}. 
							Error: " . json_last_error_msg(), Log::ERROR, 'telepost');
							continue;
						}
					}
				}
				else
				{
					if (is_array($value) || is_object($value))
					{
						Log::add("saveCustomFields: Unexpected array/object value for simple field ID {$fieldId}. 
						Converting to string.", Log::WARNING, 'telepost');
						$valueToStore = (string) json_encode($value, JSON_UNESCAPED_UNICODE);
					}
				}

				// 3. Insert Data
				$query   = $db->getQuery(true);
				$columns = ['field_id', 'item_id', 'value'];
				$values  = [
					(int) $fieldId,
					(int) $articleId,
					$valueToStore
				];

				$query->insert($db->quoteName('#__fields_values'))
					->columns($db->quoteName($columns))
					->values(implode(',', $query->bindArray($values, [ParameterType::INTEGER,
						ParameterType::INTEGER, ParameterType::STRING])));

				$db->setQuery($query);

				try
				{
					$db->execute();
					Log::add("Successfully saved custom field ID {$fieldId} (type {$fieldType}) 
					for article ID {$articleId}.",
						Log::INFO, 'telepost');
				}
				catch (\Exception $e)
				{
					Log::add("Database error saving custom field ID {$fieldId}: " . $e->getMessage(),
						Log::ERROR, 'telepost');
				}

			}

			Log::add("Finished processing all custom fields for article ID: " . $articleId,
				Log::INFO, 'telepost');

		}
		catch (\Throwable $e)
		{
			Log::add('Exception in saveCustomFields: ' . $e->getMessage(),
				Log::CRITICAL, 'telepost');
		}
	}

}
