<?php
/*
 * @package    TelePost System Plugin
 * @version     1.0.0
 * @author      CaveDesign Studio - cavedesign.ru
 * @copyright   Copyright (c) 2025 CaveDesign Studio. All Rights Reserved.
 * @license     GNU/GPL license: https://www.gnu.org/copyleft/gpl.html
 * @link        https://cavedesign.ru/
 */

namespace Joomla\Plugin\System\Telepost\Field;

\defined('_JEXEC') or die;

use Joomla\CMS\Form\FormField;
use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Uri\Uri;

/**
 * Form field for managing the Telegram webhook.
 *
 * @since  1.0.0
 */
class TelepostWebhookButtonField extends FormField
{
	/**
	 * The form field type.
	 * @var    string
	 * @since  1.0.0
	 */
	protected $type = 'TelepostWebhookButton';

	/**
	 * Method to get the field input markup.
	 *
	 * @return  string
	 *
	 * @since   1.0.0
	 */
	protected function getInput(): string
	{
		$debugMode = (int) $this->form->getValue('params.debug_mode');
		$token     = $this->form->getValue('bot_token', 'params');

		if (empty($token))
		{
			return '<div class="alert alert-info">' . Text::_('PLG_SYSTEM_TELEPOST_SAVE_TOKEN_FIRST') . '</div>';
		}

		try
		{
			$response = HttpFactory::getHttp()->get("https://api.telegram.org/bot{$token}/getWebhookInfo");
			$data     = json_decode($response->getBody(), true);

			if (empty($data['ok']))
			{
				$error = $data['description'] ?? 'Unknown error';
				return '<div class="alert alert-danger">' . Text::sprintf('PLG_SYSTEM_TELEPOST_API_ERROR', $error) . '</div>';
			}

			$webhookInfo  = $data['result'];
			$isWebhookSet = !empty($webhookInfo['url']);

		}
		catch (\Exception $e)
		{
			return '<div class="alert alert-danger">' . Text::sprintf('PLG_SYSTEM_TELEPOST_API_ERROR', $e->getMessage()) . '</div>';
		}

		$debugInfoHtml = '';
		if ($debugMode)
		{
			$debugInfoHtml = '<div class="clr"></div>';
			$debugInfoHtml .= '<h4>' . Text::_('PLG_SYSTEM_TELEPOST_WEBHOOK_STATUS_TITLE') . '</h4>';
			$debugInfoHtml .= '<pre style="white-space: pre-wrap; word-wrap: break-word; max-height: 300px; overflow-y: auto; padding: 10px; background-color: #f5f5f5; border: 1px solid #ccc; border-radius: 4px; margin-top: 20px">';
			$debugInfoHtml .= htmlspecialchars(json_encode($webhookInfo, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES), ENT_QUOTES, 'UTF-8');
			$debugInfoHtml .= '</pre>';
		}

		if ($isWebhookSet)
		{
			$action   = 'del';
			$btnClass = 'btn-danger';
			$label    = Text::_('PLG_SYSTEM_TELEPOST_BTN_DEL');
		}
		else
		{
			$action   = 'set';
			$btnClass = 'btn-success';
			$label    = Text::_('PLG_SYSTEM_TELEPOST_BTN_SET');
		}

		$ajaxUrl = Uri::root() . "index.php?option=com_ajax&plugin=telepost&group=system&task={$action}webhook&format=raw";

		$onclick = "
            const secretKey = document.querySelector('#jform_params_secret_key').value;
            if (!secretKey) {
                alert('" . Text::_('PLG_SYSTEM_TELEPOST_SECRET_KEY_DESC', true) . "');
                return false;
            }
            Joomla.request({
                url: '{$ajaxUrl}&key=' + encodeURIComponent(secretKey),
                method: 'POST',
                onSuccess: (r) => {
                    try {
                        const response = JSON.parse(r);
                        Joomla.renderMessages({[response.type]: [response.message]});
                        if (response.type === 'success') {
                            window.location.reload();
                        }
                    } catch (e) {
                        Joomla.renderMessages({error: [e.message, r]});
                    }
                },
                onError: (xhr) => Joomla.renderMessages({error: [xhr.statusText]})
            });
            return false;";

		$buttonHtml = '<button type="button" class="btn ' . $btnClass . '" onclick="' . htmlspecialchars($onclick, ENT_COMPAT, 'UTF-8') . '">'
			. $label . '</button>';

		return $buttonHtml . $debugInfoHtml;
	}
}