<?php
/*
 * @package    TelePost System Plugin
 * @version     1.0.0
 * @author      CaveDesign Studio - cavedesign.ru
 * @copyright   Copyright (c) 2025 CaveDesign Studio. All Rights Reserved.
 * @license     GNU/GPL license: https://www.gnu.org/copyleft/gpl.html
 * @link        https://cavedesign.ru/
 */

namespace Joomla\Plugin\System\Telepost\Field;

\defined('_JEXEC') or die;

use Joomla\CMS\Form\FormField;
use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Registry\Registry;
use Joomla\Plugin\System\Telepost\Helper\TelepostHelper;

/**
 * Form field for storing and synchronizing Telegram chat data.
 *
 * @since  1.0.0
 */
class ChatStorageField extends FormField
{
	/**
	 * The form field type.
	 * @var    string
	 * @since  1.0.0
	 */
	protected $type = 'ChatStorage';

	/**
	 * Method to get the field input markup.
	 *
	 * @return  string
	 *
	 * @since   1.0.0
	 */
	protected function getInput(): string
	{
		$pluginParams = PluginHelper::getPlugin('system', 'telepost')->params ?? null;
		$token = null;
		if ($pluginParams) {
			$registry = new Registry($pluginParams);
			$token = $registry->get('bot_token');
		}

		$storedChats = [];
		if (!empty($this->value) && is_string($this->value)) {
			$decoded = json_decode($this->value, true);
			if (is_array($decoded)) {
				$storedChats = $decoded;
			}
		}

		$apiChats = [];
		if ($token) {
			$apiChats = $this->fetchChatsFromApi($token);
		}

		$mergedChats = $apiChats + $storedChats;

		TelepostHelper::setRuntimeChats($mergedChats);

		$newValue = json_encode($mergedChats, JSON_UNESCAPED_UNICODE);

		return '<input type="hidden" name="' . $this->name . '" value="' . htmlspecialchars($newValue, ENT_COMPAT, 'UTF-8') . '">';
	}

	/**
	 * Fetches chat data from the Telegram API.
	 *
	 * @param   string  $token  The bot token.
	 *
	 * @return  array
	 *
	 * @since   1.0.0
	 */
	private function fetchChatsFromApi(string $token): array
	{
		$items = [];
		$http  = HttpFactory::getHttp();
		try
		{
			$response = $http->get('https://api.telegram.org/bot' . $token . '/getUpdates', [], 3);
			$data     = json_decode($response->body, true);

			if ($response->code === 200 && !empty($data['ok']))
			{
				foreach ($data['result'] ?? [] as $update)
				{
					$chat = $update['message']['chat'] ?? $update['channel_post']['chat'] ?? null;
					if ($chat)
					{
						$items[$chat['id']] = $this->formatChatTitle($chat);
					}
				}
			}
		}
		catch (\Throwable $e) {

		}

		return $items;
	}

	/**
	 * Formats the chat display title.
	 *
	 * @param   array  $chat  The chat data.
	 *
	 * @return  string
	 *
	 * @since   1.0.0
	 */
	private function formatChatTitle(array $chat): string
	{
		if (!empty($chat['title'])) return $chat['title'];
		$titleParts = [];
		if (!empty($chat['first_name'])) $titleParts[] = $chat['first_name'];
		if (!empty($chat['last_name'])) $titleParts[] = $chat['last_name'];
		if (!empty($chat['username'])) $titleParts[] = !empty($titleParts) ? '(@' . $chat['username'] . ')' : '@' . $chat['username'];

		return !empty($titleParts) ? implode(' ', $titleParts) : ($chat['id'] ?? 'Unknown Chat');
	}
}