<?php
/*
 * @package    TelePost System Plugin
 * @version     1.0.0
 * @author      CaveDesign Studio - cavedesign.ru
 * @copyright   Copyright (c) 2025 CaveDesign Studio. All Rights Reserved.
 * @license     GNU/GPL license: https://www.gnu.org/copyleft/gpl.html
 * @link        https://cavedesign.ru/
 */

namespace Joomla\Plugin\System\Telepost\Helper;

\defined('_JEXEC') or die;

use Joomla\CMS\Log\Log;
use Joomla\Filesystem\File;
use Joomla\Filesystem\Folder;
use Joomla\Filesystem\Path;
use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Filter\OutputFilter;

/**
 * Helper class for Telepost plugin operations.
 *
 * @since  1.0.0
 */
class TelepostHelper
{
	/**
	 * Runtime storage for chats.
	 * @var    array
	 * @since  1.0.0
	 */
	private static array $runtimeChats = [];

	/**
	 * Allowed file extensions.
	 * @var    array
	 * @since  1.0.0
	 */
	private const ALLOWED_EXTENSIONS = [
		// Images
		'jpg', 'jpeg', 'png', 'gif', 'webp',
		// Video
		'mp4', 'mov', 'avi', 'mkv', 'webm', 'mpeg', 'mpg'
	];

	/**
	 * Sets the runtime chats.
	 *
	 * @param   array  $chats  The chats array.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public static function setRuntimeChats(array $chats): void
	{
		self::$runtimeChats = $chats;
	}

	/**
	 * Gets the runtime chats.
	 *
	 * @return  array
	 *
	 * @since   1.0.0
	 */
	public static function getRuntimeChats(): array
	{
		return self::$runtimeChats;
	}

	/**
	 * Downloads a file from Telegram.
	 *
	 * @param   string  $fileId    The file ID.
	 * @param   string  $botToken  The bot token.
	 * @param   string  $savePath  The relative save path.
	 *
	 * @return  string|null
	 *
	 * @since   1.0.0
	 */
	public static function downloadTelegramFile(string $fileId, string $botToken, string $savePath): ?string
	{
		return self::downloadFromTelegram($fileId, $botToken, $savePath);
	}

	/**
	 * Downloads a video from Telegram.
	 *
	 * @param   array   $videoData  The video data.
	 * @param   string  $botToken   The bot token.
	 * @param   string  $savePath   The relative save path.
	 *
	 * @return  string|null
	 *
	 * @since   1.0.0
	 */
	public static function downloadTelegramVideo(array $videoData, string $botToken, string $savePath): ?string
	{
		$fileId   = $videoData['file_id'] ?? null;
		$fileName = $videoData['file_name'] ?? null;
		$fileSize = $videoData['file_size'] ?? null;

		return self::downloadFromTelegram($fileId, $botToken, $savePath, $fileName, $fileSize);
	}

	/**
	 * Internal method to download a file from Telegram.
	 *
	 * @param   string|null  $fileId    The file ID.
	 * @param   string       $botToken  The bot token.
	 * @param   string       $savePath  The relative save path.
	 * @param   string|null  $fileName  The file name.
	 * @param   int|null     $fileSize  The file size.
	 *
	 * @return  string|null
	 *
	 * @since   1.0.0
	 */
	private static function downloadFromTelegram(
		?string $fileId,
		string $botToken,
		string $savePath,
		?string $fileName = null,
		?int $fileSize = null
	): ?string {
		if (empty($botToken) || empty($fileId) || empty($savePath))
		{
			Log::add('Download failed: missing required parameters.', Log::ERROR, 'telepost');

			return null;
		}

		try
		{
			$http     = HttpFactory::getHttp();
			$apiUrl   = "https://api.telegram.org/bot{$botToken}/getFile?file_id=" . urlencode($fileId);
			$response = $http->get($apiUrl);

			if ($response->code !== 200)
			{
				Log::add("Telegram API /getFile failed with HTTP code: {$response->code}", Log::ERROR, 'telepost');

				return null;
			}

			$data = json_decode($response->body, true);

			if (empty($data['ok']) || empty($data['result']['file_path']))
			{
				Log::add('Telegram API /getFile returned an error: ' . ($data['description'] ?? 'Unknown error'),
					Log::ERROR, 'telepost');

				return null;
			}

			$tgFilePath = $data['result']['file_path'];

			$originalFileName = $fileName ?: basename($tgFilePath);
			$extension        = strtolower(pathinfo($originalFileName, PATHINFO_EXTENSION));

			if (empty($extension) || !in_array($extension, self::ALLOWED_EXTENSIONS, true))
			{
				Log::add(
					sprintf(
						'Security Alert: Attempted to download file with disallowed extension "%s". Filename: "%s". File ID: %s',
						$extension,
						$originalFileName,
						$fileId
					),
					Log::WARNING,
					'telepost'
				);

				return null;
			}

			$filenameWithoutExt = pathinfo($originalFileName, PATHINFO_FILENAME);
			$safeFilenameWithoutExt = OutputFilter::stringURLSafe($filenameWithoutExt);

			if (empty($safeFilenameWithoutExt))
			{
				$safeFilenameWithoutExt = md5($fileId);
			}

			$safeFileName = $safeFilenameWithoutExt . '.' . $extension;

			$relativePath = 'images/' . trim($savePath, '/') . '/' . $safeFileName;
			$fullSavePath = Path::clean(JPATH_ROOT . '/' . $relativePath);
			$saveDir      = dirname($fullSavePath);

			if (!is_dir($saveDir))
			{
				Folder::create($saveDir);
			}

			$fileSizeOnServer = $fileSize ?: ($data['result']['file_size'] ?? null);
			if ($fileSizeOnServer && file_exists($fullSavePath) && filesize($fullSavePath) === (int) $fileSizeOnServer)
			{
				Log::add("File already exists, skipping download: " . $relativePath, Log::DEBUG, 'telepost');

				return $relativePath;
			}

			$fileUrl = "https://api.telegram.org/file/bot{$botToken}/" . $tgFilePath;
			$fp      = fopen($fullSavePath, 'w');

			if ($fp === false)
			{
				Log::add("Failed to open file for writing: " . $fullSavePath, Log::ERROR, 'telepost');

				return null;
			}

			$ch = curl_init($fileUrl);
			curl_setopt($ch, CURLOPT_FILE, $fp);
			curl_setopt($ch, CURLOPT_TIMEOUT, 300);
			curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
			curl_setopt($ch, CURLOPT_USERAGENT, 'Joomla/TelepostPlugin');
			curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);

			$success   = curl_exec($ch);
			$curlError = curl_error($ch);
			$httpCode  = curl_getinfo($ch, CURLINFO_HTTP_CODE);

			curl_close($ch);
			fclose($fp);

			if ($success && $httpCode === 200)
			{
				Log::add("File downloaded successfully: " . $relativePath, Log::INFO, 'telepost');

				return $relativePath;
			}
			else
			{
				Log::add("cURL failed to download file. HTTP: $httpCode. Error: " . $curlError, Log::ERROR, 'telepost');
				if (File::exists($fullSavePath))
				{
					File::delete($fullSavePath);
				}

				return null;
			}
		}
		catch (\Throwable $e)
		{
			Log::add('Download exception: ' . $e->getMessage(), Log::CRITICAL, 'telepost');

			return null;
		}
	}
}