<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  System.RevoConverter
 * @author      Alex Revo
 * @e-mail      help@alexrevo.pw
 * @copyright   Copyright (C) 2019 Alex Revo. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Factory;
use Joomla\Utilities\ArrayHelper;
use Joomla\CMS\Language\Text;

class PlgSystemRevoconverter extends CMSPlugin
{
	/**
	 * Load the language file on instantiation.
	 *
	 * @var    boolean
	 * @since  3.1
	 */
	protected $autoloadLanguage = true;

	/**
	 * Constructor
	 *
	 * @param   object  &$subject  The object to observe
	 * @param   array   $config    An optional associative array of configuration settings.
	 *
	 * @since   1.0
	 */
	public function __construct(&$subject, $config)
	{
		parent::__construct($subject, $config);
		Factory::getLanguage()->load('plg_system_revoconverter', JPATH_ADMINISTRATOR . '/plugins/system/revoconverter');
	}

	/**
	 * Adds a toolbar button in com_content articles view
	 *
	 * @return  void
	 *
	 * @since   1.0
	 */
	public function onBeforeCompileHead()
	{
		$app = Factory::getApplication();

		if (!$app->isClient('administrator')) {
			return;
		}

		Factory::getLanguage()->load('plg_system_revoconverter', JPATH_ADMINISTRATOR . '/plugins/system/revoconverter');

		$option = $app->input->get('option');
		$view = $app->getInput()->get('view');

		if ($option === 'com_content' && $view === 'articles') {
			$this->addToolbarButton();
		}
	}

	/**
	 * Add the convert button to the toolbar (modern Joomla 5 approach)
	 *
	 * @return  void
	 *
	 * @since   1.0
	 */
	protected function addToolbarButton()
	{
		$app = Factory::getApplication();
		$doc = $app->getDocument();
		$wa = $doc->getWebAssetManager();

		$text = Text::_('PLG_SYSTEM_REVOCONVERTER_CONVERT_BUTTON');
		$task = 'articles.convert';
		$icon = 'publish';

		$buttonHtml = <<<HTML
<joomla-toolbar-button id="toolbar-convert" task="{$task}" list-selection="" style="display: none;">
    <button class="button-publish btn btn-success" type="button" disabled="">
        <span class="icon-{$icon}" aria-hidden="true"></span>
        {$text}
    </button>
</joomla-toolbar-button>
HTML;

		$doc->addCustomTag($buttonHtml);

		$js = <<<JS
document.addEventListener('DOMContentLoaded', function() {
    const convertButton = document.getElementById('toolbar-convert');
    const statusGroup = document.getElementById('toolbar-status-group');
    
    if (convertButton && statusGroup) {
        statusGroup.parentNode.insertBefore(convertButton, statusGroup.nextSibling);
        
        if (Joomla && Joomla.toolbar && Joomla.toolbar.initButton) {
            Joomla.toolbar.initButton(convertButton);
        }
        
        convertButton.style.display = '';
        setTimeout(() => {
            convertButton.style.opacity = '1';
        }, 10);
    }
    
    const style = document.createElement('style');
    style.textContent = `
        #toolbar-convert {
            opacity: 0;
            transition: opacity 0.3s ease;
            margin-inline-start: 0.75rem;
        }
        #toolbar joomla-toolbar-button:first-child {
            margin-inline-start: 0;
        }
    `;
    document.head.appendChild(style);
});
JS;

		$wa->addInlineScript($js);
	}

	/**
	 * Handle the convert task
	 *
	 * @return  void
	 *
	 * @since   1.0
	 */
	public function onAfterRender()
	{
		$app = Factory::getApplication();

		if (!$app->isClient('administrator')) {
			return;
		}

		$task = $app->input->get('task', '');

		if (strpos($task, 'convert') !== false) {
			$this->convert();
		}
	}

	/**
	 * Convert articles to Yootheme Pro format
	 *
	 * @return  void
	 *
	 * @since   1.0
	 */
	protected function convert()
	{
		$app = Factory::getApplication();
		$user = Factory::getUser();
		$ids = $app->input->get('cid', array(), 'array');
		$values = array('convert' => 0, 'skip' => 0, 'error' => 0, 'skip_ids' => array());

		foreach ($ids as $i => $id) {
			if (!$user->authorise('core.edit.state', 'com_content.article.' . (int)$id)) {
				unset($ids[$i]);
				$app->enqueueMessage(Text::_('JLIB_APPLICATION_ERROR_EDITSTATE_NOT_PERMITTED'), 'notice');
			}
		}

		if (empty($ids)) {
			$app->enqueueMessage(Text::_('JERROR_NO_ITEMS_SELECTED'), 'warning');
		} else {
			$result = $this->setConvert($ids, $values);

			$messageParts = [];

			if ($result['convert'] > 0) {
				$messageParts[] = Text::sprintf('PLG_SYSTEM_REVOCONVERTER_CONVERTED', $result['convert']);
			}

			if ($result['skip'] > 0) {
				$skipIds = !empty($result['skip_ids']) ? implode(', ', $result['skip_ids']) : '';
				$messageParts[] = Text::sprintf('PLG_SYSTEM_REVOCONVERTER_SKIPPED', $result['skip'], $skipIds);
			}

			if (!empty($result['error'])) {
				$messageParts[] = Text::sprintf('PLG_SYSTEM_REVOCONVERTER_ERRORS', $result['error']);
			}

			$message = implode(' ', $messageParts);

			if (!empty($message)) {
				$app->enqueueMessage($message, 'success');
			}
		}

		$app->redirect('index.php?option=com_content&view=articles');
	}

	/**
	 * Perform the actual conversion
	 *
	 * @param   array  $pks     Article IDs to convert
	 * @param   array  $values  Counter values
	 *
	 * @return  array  Updated counter values
	 *
	 * @since   1.0
	 */
	protected function setConvert($pks, $values)
	{
		$pks = ArrayHelper::toInteger((array)$pks);
		$db = Factory::getDbo();

		$query = $db->getQuery(true)
			->select($db->quoteName(['id', 'title', 'introtext', 'fulltext']))
			->from('#__content')
			->where('id IN (' . implode(',', $pks) . ')');

		$toconvert = $db->setQuery($query)->loadObjectList();

		if (!empty($toconvert)) {
			$object = new \stdClass();
			$updateNulls = true;

			foreach ($toconvert as $item) {
				if (!preg_match('/<!--\s?\{/', $item->fulltext)) {
					$object->id = $item->id;

					$title = trim(str_replace(["\r", "\n"], '', $item->title));
					$content = trim(str_replace(["\r", "\n"], '', $item->introtext . $item->fulltext));

					$title = htmlspecialchars_decode($title, ENT_QUOTES);
					$content = htmlspecialchars_decode($content, ENT_QUOTES);

					$children = [];

					if ($this->params->get('addbreadcrumbs', 0)) {
						$children[] = [
							'type' => 'breadcrumbs',
							'props' => ['show_current' => true, 'show_home' => true]
						];
					}

					if ($this->params->get('addtitle', 0)) {
						$children[] = [
							'type' => 'headline',
							'props' => ['title_element' => 'h1', 'content' => $title]
						];
					}

					$children[] = [
						'type' => 'html',
						'props' => ['content' => $content]
					];

					$layout = [
						'type' => 'layout',
						'children' => [
							[
								'type' => 'section',
								'props' => [
									'image_position' => 'center-center',
									'style' => 'default',
									'title_breakpoint' => 'xl',
									'title_position' => 'top-left',
									'title_rotation' => 'left',
									'vertical_align' => 'middle',
									'width' => 'default'
								],
								'children' => [
									[
										'type' => 'row',
										'children' => [
											[
												'type' => 'column',
												'props' => [
													'image_position' => 'center-center',
													'position_sticky_breakpoint' => 'm'
												],
												'children' => $children
											]
										],
										'props' => []
									]
								]
							]
						],
						'version' => '4.5.17',
						'props' => []
					];

					$encodedLayout = json_encode($layout, JSON_UNESCAPED_UNICODE | JSON_HEX_QUOT);
					$object->fulltext = '<!-- ' . $encodedLayout . ' -->';

					if ($db->updateObject('#__content', $object, 'id', $updateNulls)) {
						$values['convert']++;
					} else {
						$values['error'] .= $item->id . ';';
					}
				} else {
					$values['skip_ids'][] = $item->id;
					$values['skip']++;
				}
			}
		}

		return $values;
	}
}
